//
// gqgmc.go
// Copyright (C) 2017 kevin <kevin@phrye.com>
//
// Distributed under terms of the GPL license.
//

package geiger

import (
	"errors"
	"fmt"
	"time"

	"github.com/tarm/serial"
)

// The Ideal and Low voltage threshholds
const (
	VoltageIdeal  = 98
	VoltageTooLow = 75
)

const (
	powerOnOff                    = 0
	alarmOnOff                    = 1
	speakerOnOff                  = 2
	graphicModeOnOff              = 3
	backlightTimeoutSeconds       = 4
	idleTitleDisplayMode          = 5
	alarmCPMValue                 = 6
	calibrationCPM0               = 8
	calibrationSvUc0              = 10
	calibrationCPM1               = 14
	calibrationSvUc1              = 16
	calibrationCPM2               = 20
	calibrationSvUc2              = 22
	idleDisplayMode               = 26
	alarmValueuSvUc               = 27
	alarmType                     = 31
	saveDataType                  = 32
	swivelDisplay                 = 33
	zoom                          = 34
	dataSaveAddress               = 38
	dataReadAddress               = 41
	nPowerSavingMode              = 44
	nSensitivityMode              = 45
	nCounterDelay                 = 46
	nVoltageOffset                = 48
	maxCPM                        = 49
	nSensitivityAutoModeThreshold = 51
	saveDate                      = 52
	saveTime                      = 55
	maxBytes                      = 58
)

var configBytes = map[int]int{
	powerOnOff:              1,
	alarmOnOff:              1,
	speakerOnOff:            1,
	graphicModeOnOff:        1,
	backlightTimeoutSeconds: 1,
	idleTitleDisplayMode:    1,
	alarmCPMValue:           2,
	calibrationCPM0:         2,
	calibrationSvUc0:        4,
	calibrationCPM1:         2,
	calibrationSvUc1:        4,
	calibrationCPM2:         2,
	calibrationSvUc2:        4,
	idleDisplayMode:         1,
	alarmValueuSvUc:         4,
	alarmType:               1,
	saveDataType:            1,
	swivelDisplay:           1,
	zoom:                    4,
	dataSaveAddress:         3,
	dataReadAddress:         3,
	nPowerSavingMode:        1,
	nSensitivityMode:        1,
	nCounterDelay:           2,
	nVoltageOffset:          1,
	maxCPM:                  2,
	nSensitivityAutoModeThreshold: 1,
	saveDate:                      3,
	saveTime:                      3,
	maxBytes:                      1,
}

const (
	saveOff = iota
	saveCPS
	saveCPM
	saveCPH
	saveMax
)

const (
	historyDataMaxSize = 0x1000
	historyAddrMaxSize = 0x10000
	forFirmware        = 2.23
	nvmSize            = 256
)

const (
	cmdGetSerial    = "<GETSERIAL>>"
	cmdGetVersion   = "<GETVER>>"
	cmdGetVoltage   = "<GETVOLT>>"
	cmdGetCPM       = "<GETCPM>>"
	cmdGetCPS       = "<GETCPS>>"
	cmdGetCfg       = "<GETCFG>>"
	cmdEraseCfg     = "<ECFG>>"
	cmdUpdateCfg    = "<CFGUPDATE>>"
	cmdTurnOnCPS    = "<HEARTBEAT1>>"
	cmdTurnOffCPS   = "<HEARTBEAT0>>"
	cmdTurnOffPwr   = "<POWEROFF>>"
	cmdFactoryReset = "<FACTORYRESET>>"
	cmdReboot       = "<REBOOT>>"
	cmdGetTime      = "<GETDATETIME>>"
)

// GQGMCCounter is a GQ GMC Counter
type GQGMCCounter struct {
	port   *serial.Port
	config *serial.Config
}

// NewGQGMC creates a new GQGMC Counter instance
func NewGQGMC(c Config) (*GQGMCCounter, error) {
	cfg := serial.Config{
		Name:        c.Device,
		Baud:        57600,
		ReadTimeout: 500 * time.Millisecond,
	}
	p, err := serial.OpenPort(&cfg)
	if err != nil {
		return nil, err
	}
	//getConfigurationData()
	return &GQGMCCounter{port: p, config: &cfg}, nil
}

// Clear clears out any remaining data
func (gc *GQGMCCounter) Clear() error {
	// Read up to 10 chars until nothing comes back.
	// error otherwise.
	for i := 0; i < 10; i++ {
		if _, err := gc.readCmd(1); err != nil {
			break
		}
	}
	return nil
}

// Version gets the version of the device
func (gc *GQGMCCounter) Version() (string, error) {
	ver, err := gc.communicate(cmdGetVersion, 14)
	return string(ver), err
}

// SerialNum gets the serial number of the device
func (gc *GQGMCCounter) SerialNum() (string, error) {
	serStr := ""

	ser, err := gc.communicate(cmdGetSerial, 7)
	if err == nil {
		for _, b := range ser {
			serStr += fmt.Sprintf("%02X", b)
		}
	}
	return serStr, err
}

func (gc *GQGMCCounter) getReading(what string) (uint16, error) {
	buf, err := gc.communicate(what, 2)
	if err != nil {
		return 0, err
	}
	reading := ((uint16(buf[0]) << 8) & 0x3f00)
	reading |= (uint16(buf[1]) & 0x00ff)
	return reading, nil
}

// GetCPM returns CPM
func (gc *GQGMCCounter) GetCPM() (uint16, error) {
	return gc.getReading(cmdGetCPM)
}

// GetCPS returns CPS
func (gc *GQGMCCounter) GetCPS() (uint16, error) {
	return gc.getReading(cmdGetCPS)
}

// Volts returns current battery voltage
func (gc *GQGMCCounter) Volts() (int16, error) {
	// Do this differently - for 9.6 return 96. And if not supported,
	// Return -1.
	// Public method to read voltage value of battery. The GQ GMC returns
	// a single byte whose integer value converted to a float divided by 10
	// equals the battery voltage. For example, 0x60 = 96 converts to 9.6 Volts.
	// The ideal value is 9.8 volts. So practically speaking, we should not
	// expect to see a value higher than 100. The command is get_voltage_cmd
	// (see GQ GMC COMMANDS above). If the voltage falls below 7.5V, GQ LLC
	// says the geiger counter cannot be expected to operate properly.
	volts, err := gc.communicate(cmdGetVoltage, 1)
	if err != nil {
		return 0, err
	}
	return int16(volts[0]), err
}

// GetHistoryData Should return history data but is unimplemented for now
func (gc *GQGMCCounter) GetHistoryData() {
	// It's not recommended to use this so blank for now.
	return
}

// TurnOnCPS turns on CPS collection
func (gc *GQGMCCounter) TurnOnCPS() error {
	// turnOnCPS is public method to enable automatic reporting of the
	// count per second (CPS) value. First, I would say don't use this
	// command. Since the returned data has no protocol, that is, there
	// is no start/stop marker, no identification, no nothing but a
	// sequence of bytes, any other command issued while CPS is turned on
	// will take extraordinary effort not to confuse its returned data
	// with the CPS data. To handle it correctly, you would have to
	// wait for the CPS data to be returned, and then issue the command.
	// This would be most safely done by creating a new thread to
	// read the CPS and using a mutex to signal opportunity to issue a
	// separate command. The command is turn_on_cps_cmd
	// (see GQ GMC COMMANDS above). The returned data is always two
	// bytes so that samples > 255 can be reported (even though unlikely).
	gc.sendCmd(cmdTurnOnCPS)
	return nil
}

// TurnOffCPS turns off CPS collection
func (gc *GQGMCCounter) TurnOffCPS() error {
	gc.sendCmd(cmdTurnOffCPS)
	gc.Clear()
	return nil
}

// GetAutoCPS gets a reading once auto CPS is turned on
func (gc *GQGMCCounter) GetAutoCPS() (uint16, error) {
	buf, err := gc.readCmd(2)
	if err != nil {
		return 0, err
	}
	cps := ((uint16(buf[0]) << 8) & 0x3f00)
	cps |= (uint16(buf[1]) & 0x00ff)
	return cps, nil
}

// TurnOffPower turns the device off
func (gc *GQGMCCounter) TurnOffPower() {
	gc.sendCmd(cmdTurnOffPwr)
	return
}

// GetConfiguration reads configuration data
func (gc *GQGMCCounter) GetConfiguration() {
	// Issue command to get configuration and read returned data.
	// communicate(get_cfg_cmd, reinterpret_cast<char *>(&mCFG_Data),
}

// SetConfiguration writes configuration data
func (gc *GQGMCCounter) SetConfiguration() {
	// See the ConfigurationData functions in gqgmc.cc
}

// ResetConfiguration resets to factory default
func (gc *GQGMCCounter) ResetConfiguration() {
	//uint32_t     retsize = 1;
	//char         ret_char[retsize+1];
	//communicate(erase_cfg_cmd, ret_char, retsize);
}

// SetTime sets the time
func (gc *GQGMCCounter) SetTime(t time.Time) {
	//command: <SETDATETIME[YYMMDDHHMMSS]>>
	//Firmware supported: GMC-280, GMC-300 Re.3.00 or later

	cmd := make([]byte, 13)
	var timeCmds = []struct {
		cmd  string
		unit int
	}{
		{"<SETDATEYY", t.Year()},
		{"<SETDATEMM", int(t.Month())},
		{"<SETDATEDD", t.Day()},
		{"<SETTIMEHH", t.Hour()},
		{"<SETTIMEMM", t.Minute()},
		{"<SETTIMESS", t.Second()},
	}

	for _, c := range timeCmds {
		copy(cmd[:], c.cmd)
		cmd[10] = uint8(c.unit)
		copy(cmd[11:], ">>")
		gc.port.Write(cmd)
		gc.readCmd(1)
	}
}

// GetTime gets the time
func (gc *GQGMCCounter) GetTime() (time.Time, error) {
	b, err := gc.communicate(cmdGetTime, 7)
	if err != nil {
		return time.Unix(0, 0), err
	}
	t := time.Date(int(b[0])+2000, time.Month(b[1]), int(b[2]),
		int(b[3]), int(b[4]), int(b[5]), 0, time.Local)
	return t, nil
}

// FactoryReset does a factory reset
func (gc *GQGMCCounter) FactoryReset() {
	gc.sendCmd(cmdFactoryReset)
	return
}

// Reboot reboots the device
func (gc *GQGMCCounter) Reboot() {
	gc.sendCmd(cmdReboot)
	return
}

func (gc *GQGMCCounter) communicate(cmd string, length uint32) ([]byte, error) {
	gc.Clear()
	if len(cmd) > 0 {
		gc.sendCmd(cmd)
	}
	if length != 0 {
		return gc.readCmd(length)
	}
	return nil, nil
}

func (gc *GQGMCCounter) sendCmd(cmd string) {
	gc.port.Write([]byte(cmd))
	return
}

func (gc *GQGMCCounter) readCmd(length uint32) ([]byte, error) {
	buf := make([]byte, length)
	n, err := gc.port.Read(buf)
	if err != nil {
		return nil, err
	}
	if uint32(n) != length {
		return nil, errors.New("Short read")
	}
	return buf, nil
}
