//
// gqgmc.go
// Copyright (C) 2017 kevin <kevin@phrye.com>
//
// Distributed under terms of the GPL license.
//

package geiger

import (
	"errors"
	"fmt"
	"strconv"
	"time"

	"github.com/tarm/serial"
)

// The Ideal and Low voltage threshholds
const (
	VoltageIdeal  = 98
	VoltageTooLow = 75
)

const (
	cmdGetSerial    = "<GETSERIAL>>"    // GMC-280, GMC-300 Re.2.11
	cmdGetVersion   = "<GETVER>>"       // GMC-280, GMC-300 Re.2.0x, Re.2.10
	cmdGetVoltage   = "<GETVOLT>>"      // GMC-280, GMC-300 Re.2.0x, Re.2.10
	cmdGetCPM       = "<GETCPM>>"       // GMC-280, GMC-300 Re.2.0x, Re.2.10
	cmdGetCPS       = "<GETCPS>>"       // GMC-280, GMC-300 Re.2.0x, Re.2.10
	cmdGetCfg       = "<GETCFG>>"       // GMC-280, GMC-300 Re.2.10
	cmdEraseCfg     = "<ECFG>>"         // GMC-280, GMC-300 Re.2.10
	cmdUpdateCfg    = "<CFGUPDATE>>"    // GMC-280, GMC-300 Re.2.20
	cmdTurnOnCPS    = "<HEARTBEAT1>>"   // GMC-280, GMC-300 Re.2.10
	cmdTurnOffCPS   = "<HEARTBEAT0>>"   // Re.2.10
	cmdTurnOffPwr   = "<POWEROFF>>"     // GMC-280, GMC-300 Re.2.11
	cmdTurnOnPwr    = "<POWERON>>"      // GMC-280, GMC-300 Re.3.10
	cmdFactoryReset = "<FACTORYRESET>>" // GMC-280, GMC-300 Re.3.00
	cmdReboot       = "<REBOOT>>"       // GMC-280, GMC-300 Re.3.00
	cmdGetTime      = "<GETDATETIME>>"  // GMC-280, GMC-300 Re.3.00
	cmdGetTemp      = "<GETTEMP>>"      // GMC-320 Re.3.01
	cmdGetGyro      = "<GETGYRO>>"      // GMC-320 Re.3.01
)

// GQGMCCounter is a GQ GMC Counter
type GQGMCCounter struct {
	port   *serial.Port
	config *serial.Config
	version,
	model string
}

// NewGQGMC creates a new GQGMC Counter instance
func NewGQGMC(c Config) (*GQGMCCounter, error) {
	var gc GQGMCCounter
	var v []byte

	gc.config = &serial.Config{
		Name:        c.Device,
		Baud:        57600,
		ReadTimeout: 500 * time.Millisecond,
	}
	p, err := serial.OpenPort(gc.config)
	if err != nil {
		return nil, err
	}
	gc.port = p
	v, err = gc.communicate(cmdGetVersion, 14)
	gc.model = string(v[:7])
	gc.version = string(v[7:])
	//getConfigurationData()
	return &gc, nil
}

// Clear clears out any remaining data
func (gc *GQGMCCounter) Clear() error {
	// Read up to 10 chars until nothing comes back.
	// error otherwise.
	for i := 0; i < 10; i++ {
		if _, err := gc.readCmd(1); err != nil {
			break
		}
	}
	return nil
}

// Version gets the version of the device
func (gc *GQGMCCounter) Version() string {
	return gc.version
}

// Model gets the model of the device
func (gc *GQGMCCounter) Model() string {
	return gc.model
}

// SerialNum gets the serial number of the device
func (gc *GQGMCCounter) SerialNum() (string, error) {
	serStr := ""

	ser, err := gc.communicate(cmdGetSerial, 7)
	if err == nil {
		for _, b := range ser {
			serStr += fmt.Sprintf("%02X", b)
		}
	}
	return serStr, err
}

func (gc *GQGMCCounter) getReading(what string) (uint16, error) {
	buf, err := gc.communicate(what, 2)
	if err != nil {
		return 0, err
	}
	reading := ((uint16(buf[0]) << 8) & 0x3f00)
	reading |= (uint16(buf[1]) & 0x00ff)
	return reading, nil
}

// GetCPM returns CPM
func (gc *GQGMCCounter) GetCPM() (uint16, error) {
	return gc.getReading(cmdGetCPM)
}

// GetCPS returns CPS
func (gc *GQGMCCounter) GetCPS() (uint16, error) {
	return gc.getReading(cmdGetCPS)
}

// Volts returns current battery voltage
func (gc *GQGMCCounter) Volts() (int16, error) {
	// Do this differently - for 9.6 return 96. And if not supported,
	// Return -1.
	// Public method to read voltage value of battery. The GQ GMC returns
	// a single byte whose integer value converted to a float divided by 10
	// equals the battery voltage. For example, 0x60 = 96 converts to 9.6 Volts.
	// The ideal value is 9.8 volts. So practically speaking, we should not
	// expect to see a value higher than 100. The command is get_voltage_cmd
	// (see GQ GMC COMMANDS above). If the voltage falls below 7.5V, GQ LLC
	// says the geiger counter cannot be expected to operate properly.
	volts, err := gc.communicate(cmdGetVoltage, 1)
	if err != nil {
		return 0, err
	}
	return int16(volts[0]), err
}

// GetHistoryData Should return history data but is unimplemented for now
func (gc *GQGMCCounter) GetHistoryData() {
	// It's not recommended to use this so blank for now.
	return
}

// TurnOnCPS turns on CPS collection
func (gc *GQGMCCounter) TurnOnCPS() error {
	gc.sendCmd(cmdTurnOnCPS)
	return nil
}

// TurnOffCPS turns off CPS collection
func (gc *GQGMCCounter) TurnOffCPS() error {
	gc.sendCmd(cmdTurnOffCPS)
	gc.Clear()
	return nil
}

// GetAutoCPS gets a reading once auto CPS is turned on
func (gc *GQGMCCounter) GetAutoCPS() (uint16, error) {
	buf, err := gc.readCmd(2)
	if err != nil {
		return 0, err
	}
	cps := ((uint16(buf[0]) << 8) & 0x3f00)
	cps |= (uint16(buf[1]) & 0x00ff)
	return cps, nil
}

// TurnOnPower turns the device on
func (gc *GQGMCCounter) TurnOnPower() {
	gc.sendCmd(cmdTurnOnPwr)
	return
}

// TurnOffPower turns the device off
func (gc *GQGMCCounter) TurnOffPower() {
	gc.sendCmd(cmdTurnOffPwr)
	return
}

// GetConfiguration reads configuration data
func (gc *GQGMCCounter) GetConfiguration() {
	cfg, err := gc.communicate(cmdGetCfg, 256)
	if err != nil {
		return
	}
	fmt.Printf("%+v\n", cfg)
}

// SetConfiguration writes configuration data
func (gc *GQGMCCounter) SetConfiguration() {
	// See the ConfigurationData functions in gqgmc.cc
}

// SetTime sets the time
func (gc *GQGMCCounter) SetTime(t time.Time) {
	if !gc.supportedModels([]string{"GMC-280", "GMC-300"}) {
		return
	}
	if gc.versionLT("Re 2.23") {
		return
	}
	if gc.versionLT("Re 3.30") {
		gc.setTimeParts(t)
	}
	gc.setTimeAll(t)
}

func (gc *GQGMCCounter) setTimeParts(t time.Time) {
	cmd := make([]byte, 13)
	var timeCmds = []struct {
		cmd  string
		unit int
	}{
		{"<SETDATEYY", t.Year()},
		{"<SETDATEMM", int(t.Month())},
		{"<SETDATEDD", t.Day()},
		{"<SETTIMEHH", t.Hour()},
		{"<SETTIMEMM", t.Minute()},
		{"<SETTIMESS", t.Second()},
	}

	for _, c := range timeCmds {
		copy(cmd[:], c.cmd)
		cmd[10] = uint8(c.unit)
		copy(cmd[11:], ">>")
		fmt.Printf("%s: %+v\n", c.cmd, cmd)
		//gc.port.Write(cmd)
		//gc.readCmd(1)
	}
}

func (gc *GQGMCCounter) setTimeAll(t time.Time) {
	cmd := make([]byte, 20)
	copy(cmd[:], "<SETDATETIME")
	cmd[12] = uint8(t.Year())
	cmd[13] = uint8(int(t.Month()))
	cmd[14] = uint8(t.Day())
	cmd[15] = uint8(t.Hour())
	cmd[16] = uint8(t.Minute())
	cmd[17] = uint8(t.Second())
	copy(cmd[18:], ">>")
	fmt.Printf("setTimeAll: %+v\n", cmd)
	//gc.port.Write(cmd)
	//gc.readCmd(1)
}

// GetTime gets the time
func (gc *GQGMCCounter) GetTime() (time.Time, error) {
	b, err := gc.communicate(cmdGetTime, 7)
	if err != nil {
		return time.Unix(0, 0), err
	}
	t := time.Date(int(b[0])+2000, time.Month(b[1]), int(b[2]),
		int(b[3]), int(b[4]), int(b[5]), 0, time.Local)
	return t, nil
}

// GetTemp gets the temp
func (gc *GQGMCCounter) GetTemp() (float64, error) {
	t, err := gc.communicate(cmdGetTemp, 4)
	if err != nil {
		return 0, err
	}

	var temp float64
	temp, err = strconv.ParseFloat(fmt.Sprintf("%d.%d", uint8(t[0]), uint8(t[1])), 64)
	if err != nil {
		return 0, err
	}
	if t[2] != 0 {
		temp = -temp
	}
	return temp, nil
}

// GetGyro gets the position in space
func (gc *GQGMCCounter) GetGyro() (int16, int16, int16, error) {
	buf, err := gc.communicate(cmdGetGyro, 7)
	if err != nil {
		return 0, 0, 0, err
	}
	x := (int16(buf[0]) << 8)
	x |= (int16(buf[1]) & 0x00ff)
	y := (int16(buf[0]) << 8)
	y |= (int16(buf[1]) & 0x00ff)
	z := (int16(buf[0]) << 8)
	z |= (int16(buf[1]) & 0x00ff)
	return x, y, z, nil
}

// FactoryReset does a factory reset
func (gc *GQGMCCounter) FactoryReset() {
	gc.sendCmd(cmdFactoryReset)
	return
}

// Reboot reboots the device
func (gc *GQGMCCounter) Reboot() {
	gc.sendCmd(cmdReboot)
	return
}

func (gc *GQGMCCounter) supportedModels(models []string) bool {
	for _, model := range models {
		if model == gc.model {
			return true
		}
	}
	return false
}

func (gc *GQGMCCounter) versionLT(version string) bool {
	return gc.version < version
}

func (gc *GQGMCCounter) communicate(cmd string, length uint32) ([]byte, error) {
	gc.Clear()
	if len(cmd) > 0 {
		gc.sendCmd(cmd)
	}
	if length != 0 {
		return gc.readCmd(length)
	}
	return nil, nil
}

func (gc *GQGMCCounter) sendCmd(cmd string) {
	gc.port.Write([]byte(cmd))
	return
}

func (gc *GQGMCCounter) readCmd(length uint32) ([]byte, error) {
	buf := make([]byte, length)
	n, err := gc.port.Read(buf)
	if err != nil {
		return nil, err
	}
	if uint32(n) != length {
		return nil, errors.New("Short read")
	}
	return buf, nil
}
