//
// main.go
// Copyright (C) 2017 kevin <kevin@ie.suberic.net>
//
// Distributed under terms of the MIT license.
//

package main

import (
	"flag"
	"html/template"
	"log"
	"net/http"
	"path"

	"github.com/prometheus/client_golang/prometheus/promhttp"
	"gitlab.com/lyda/gqgmc/devices/geiger"
)

var addr = flag.String("listen-address", ":8080", "Address for HTTP requests.")
var device = flag.String("device", "/dev/gqgmc", "Device for Geiger Counter.")
var model = flag.String("model", "gqgmc", "Model of Geiger Counter.")
var templateDir = flag.String("template-dir", "templates", "Template directory.")
var staticDir = flag.String("static-dir", "static", "Static files directory.")

type indexPage struct {
	Model   string
	Version string
	Serial  string
	Volts   int16
	CPM     uint16
}

var gc geiger.Counter
var indexPg indexPage

func indexHandler(w http.ResponseWriter, r *http.Request) {
	indexPg.CPM, _ = gc.GetCPM()
	indexPg.Volts, _ = gc.Volts()
	t, err := template.ParseFiles(path.Join(*templateDir, "index.html"))
	if err != nil {
		log.Printf("Template error: %s\n", err)
	}
	t.Execute(w, &indexPg)
}

func staticHandler(w http.ResponseWriter, r *http.Request) {
	staticFile := path.Join(*staticDir, path.Base(r.URL.Path))
	http.ServeFile(w, r, staticFile)
}

func main() {
	flag.Parse()

	gc, _ = geiger.New(geiger.Config{Model: *model, Device: *device})
	indexPg.Model = gc.Model()
	indexPg.Version = gc.Version()
	indexPg.Serial = gc.Serial()
	http.HandleFunc("/", indexHandler)
	http.HandleFunc("/favicon.ico", staticHandler)
	http.HandleFunc("/robots.txt", staticHandler)
	http.HandleFunc("/humans.txt", staticHandler)
	http.Handle("/metrics", promhttp.Handler())
	http.Handle("/static", http.StripPrefix("/static/", http.FileServer(http.Dir(*staticDir))))
	log.Fatal(http.ListenAndServe(*addr, nil))
}
