// Package bar handles the main program for i3going-on.
// Copyright (C) 2022 Kevin Lyda <kevin@lyda.ie>
package bar

import (
	"fmt"
	"os/exec"
	"path/filepath"
	"strings"
	"time"

	"github.com/fsnotify/fsnotify"
	"github.com/spf13/cobra"
	"github.com/spf13/viper"
	"gitlab.ie.suberic.net/kevin/i3going-on/modules"
	"golang.org/x/sys/unix"
)

// Run is essentially the main program.
func Run(cmd *cobra.Command, args []string) {
	// Get our config file.
	viper.BindPFlags(cmd.Flags())
	cfg := readConfig(viper.GetString("config"))

	// Watch our config file for changes.
	watcher, err := fsnotify.NewWatcher()
	cobra.CheckErr(err)
	configDir := filepath.Dir(viper.GetString("config"))
	watcher.Add(configDir)
	defer watcher.Close()

	// Handle the user clicking things.
	clicks := make(chan modules.Click, 1)
	go handleCommands(clicks)

	fmt.Printf(`{ "version": 1, "stop_signal": %d, "cont_signal": %d, "click_events": true }`+"\n",
		unix.SignalNum("SIGSTOP"), unix.SignalNum("SIGCONT"))
	fmt.Printf("[\n[{\"full_text\": \"Starting i3going-on...\"}]\n")
	time.Sleep(2 * time.Second)
	for {
		// Render the status line.
		line := make([]string, 0, 5)
		for _, module := range cfg.Modules {
			line = append(line, module.Render())
		}
		fmt.Printf(",\n[%s]", strings.Join(line, ","))

		// Wait for the refresh time, a click or a config file change.
		select {
		case click := <-clicks:
			found := false
			for _, module := range cfg.Modules {
				if click.Name == module.Name {
					found = true
					if module.OnClick != "" {
						cmd := exec.Command("/bin/sh", "-c", module.OnClick)
						cmd.Run()
					}
				}
			}
			if !found {
				fmt.Printf(`,[{"full_text": "You clicked '%s'. Huh?"}]%c`, click.Name, '\n')
				time.Sleep(5 * time.Second)
			}
		case configEvent := <-watcher.Events:
			if configEvent.Name == viper.GetString("config") && !configEvent.Has(fsnotify.Remove) {
				cfg = readConfig(viper.GetString("config"))
			}
		case <-time.After(time.Duration(cfg.Refresh) * time.Second):
		}
	}
}
