// Package modules implements all the modules.
// Copyright (C) 2022 Kevin Lyda <kevin@lyda.ie>
package modules

import (
	"fmt"
	"strings"

	"gopkg.in/yaml.v3"
)

// ParamsInterface is the interface for module paramaters.
type ParamsInterface interface {
	SetDefaults()
	Name() string
	OnClick() string
	Render() string
}

// Module defines a module.
type Module struct {
	Module  string          `yaml:"module"`
	Name    string          `yaml:"name"`
	OnClick string          `yaml:"on-click"`
	Params  ParamsInterface `yaml:"-"`
}

// M is an intermediate type to avoid name collisions.
type M Module

// Params is the structure for module specific params.
type Params struct {
	*M     `yaml:",inline"`
	Params yaml.Node `yaml:"params"`
}

// UnmarshalYAML is a helper program to unmarshall the specific params.
func (m *Module) UnmarshalYAML(node *yaml.Node) error {
	params := &Params{M: (*M)(m)}
	if err := node.Decode(params); err != nil {
		return err
	}

	if strings.ContainsAny(m.Name, "{}") {
		return fmt.Errorf("module name field can't contain '{' or '}' characters")
	}
	if m.Name == "" {
		m.Name = m.Module
	}
	switch params.Module {
	case "date":
		m.Params = NewDate(m)
	case "text":
		m.Params = NewText(m)
	case "battery":
		m.Params = NewBattery(m)
	case "interfaces":
		m.Params = NewInterfaces(m)
	default:
		return fmt.Errorf("module '%s' is unknown", params.Module)
	}
	return params.Params.Decode(m.Params)
}

func (m *Module) Render() string {
	return m.Params.Render()
}

func (m *Module) SetDefaults() {
	m.Params.SetDefaults()
}
