// Package folders are all the routines and sql for managing folders.
package folders

import (
	"embed"
	"errors"
	"os"
	"path"

	"github.com/adrg/xdg"
	"github.com/golang-migrate/migrate/v4"
	"github.com/golang-migrate/migrate/v4/source/iofs"
	"github.com/jmoiron/sqlx"

	// Included to connect to sqlite.
	_ "github.com/golang-migrate/migrate/v4/database/sqlite"
	_ "modernc.org/sqlite"
)

//go:embed sql/*.sql
var fs embed.FS

// Store is the store for folders.
type Store struct {
	user string
	db   *sqlx.DB
}

// Open opens the folders database.
func Open(user string) (*Store, error) {
	fdir := path.Join(xdg.DataHome, "BULLETIN")
	err := os.MkdirAll(fdir, 0700)
	if err != nil {
		return nil, errors.New("bulletin directory problem")
	}
	fdb := path.Join(fdir, "bboard.db")

	// Run db migrations if needed.
	sqldir, err := iofs.New(fs, "sql")
	if err != nil {
		return nil, err
	}
	m, err := migrate.NewWithSourceInstance("iofs", sqldir, "sqlite://"+fdb+"?_pragma=foreign_keys(1)")
	if err != nil {
		return nil, err
	}
	err = m.Up()
	if err != nil && err != migrate.ErrNoChange {
		return nil, err
	}
	m.Close()

	store := &Store{user: user}
	store.db, err = sqlx.Connect("sqlite", "file://"+fdb+"?_pragma=foreign_keys(1)")
	if err != nil {
		return nil, errors.New("bulletin database problem")
	}
	return store, nil
}

// Close closes the db backing the store.
func (fstore *Store) Close() {
	fstore.db.Close()
}
