CREATE TABLE users (
  login       VARCHAR(12)  NOT NULL PRIMARY KEY,
  name        VARCHAR(53)  NOT NULL,
  admin       INT          DEFAULT 0 NOT NULL,
  moderator   INT          DEFAULT 0 NOT NULL,
  alert       INT          NOT NULL DEFAULT 0,  --- 0=no, 1=brief, 2=readnew
  disabled    INT          DEFAULT 0 NOT NULL,
  last_login  TIMESTAMP    DEFAULT CURRENT_TIMESTAMP NOT NULL,
  create_at   TIMESTAMP    DEFAULT CURRENT_TIMESTAMP NOT NULL,
  update_at   TIMESTAMP    DEFAULT CURRENT_TIMESTAMP NOT NULL
) WITHOUT ROWID;

CREATE TRIGGER users_after_update_update_at
  AFTER UPDATE ON users FOR EACH ROW
  WHEN NEW.update_at = OLD.update_at    --- avoid infinite loop
BEGIN
  UPDATE users SET update_at=CURRENT_TIMESTAMP WHERE login=NEW.login;
END;

INSERT INTO users (login, name, admin)
       VALUES ('SYSTEM', 'System User', 1);

CREATE TRIGGER users_before_update_protect
  AFTER UPDATE ON users FOR EACH ROW
  WHEN OLD.login = 'SYSTEM' AND (NEW.login != OLD.login OR NEW.admin != 1)
BEGIN
  SELECT RAISE (ABORT, 'SYSTEM user is protected');
END;

CREATE TRIGGER users_before_delete_protect
  BEFORE DELETE on users FOR EACH ROW
  WHEN OLD.login = 'SYSTEM'
BEGIN
  SELECT RAISE (ABORT, 'SYSTEM user is protected');
END;

CREATE TABLE folders (
  name        VARCHAR(25)  NOT NULL PRIMARY KEY,
  always      INT          DEFAULT 0 NOT NULL,
  brief       INT          DEFAULT 0 NOT NULL,
  description VARCHAR(53)  DEFAULT 0 NOT NULL,
  notify      INT          DEFAULT 0 NOT NULL,
  readnew     INT          DEFAULT 0 NOT NULL,
  shownew     INT          DEFAULT 0 NOT NULL,
  system      INT          DEFAULT 0 NOT NULL,
  expire      INT          DEFAULT 14 NOT NULL,
  visibility  TEXT         DEFAULT 'public' NOT NULL,
  create_at   TIMESTAMP    DEFAULT CURRENT_TIMESTAMP NOT NULL,
  update_at   TIMESTAMP    DEFAULT CURRENT_TIMESTAMP NOT NULL
) WITHOUT ROWID;

CREATE TRIGGER folders_before_insert_validate
  BEFORE INSERT on folders
BEGIN
  SELECT
    CASE
      WHEN NEW.name != UPPER(NEW.name) OR NEW.name GLOB '*[^A-Z0-9_-]*' THEN
        RAISE (ABORT, 'Invalid folder name')
    END;
END;

CREATE TRIGGER folders_before_update_validate
  BEFORE UPDATE on folders
BEGIN
  SELECT
    CASE
      WHEN NEW.name != UPPER(NEW.name) OR NEW.name GLOB '*[^A-Z0-9_-]*' THEN
        RAISE (ABORT, 'Invalid folder name')
      WHEN OLD.name = 'GENERAL' AND OLD.name != NEW.name THEN
        RAISE (ABORT, 'GENERAL folder is protected')
    END;
END;

CREATE TRIGGER folders_after_update_update_at
  AFTER UPDATE ON folders FOR EACH ROW
    WHEN NEW.update_at = OLD.update_at    --- avoid infinite loop
BEGIN
  UPDATE folders SET update_at=CURRENT_TIMESTAMP WHERE name=NEW.name;
END;

CREATE TRIGGER folders_before_delete_protect
  BEFORE DELETE on folders FOR EACH ROW
  WHEN OLD.name = 'GENERAL'
BEGIN
  SELECT RAISE (ABORT, 'GENERAL folder is protected');
END;

INSERT INTO folders (name, description, system, shownew, owner)
       VALUES ('GENERAL', 'Default general bulletin folder.', 1, 1, 'SYSTEM');

CREATE TABLE owners (
  folder      VARCHAR(25)  REFERENCES folders(name) ON DELETE CASCADE ON UPDATE CASCADE,
  owner       VARCHAR(25)  REFERENCES users(login) ON UPDATE CASCADE,
  create_at   TIMESTAMP    DEFAULT CURRENT_TIMESTAMP NOT NULL,
  update_at   TIMESTAMP    DEFAULT CURRENT_TIMESTAMP NOT NULL,
  PRIMARY KEY (folder, owner)
) WITHOUT ROWID;

CREATE TRIGGER owners_after_update_update_at
  AFTER UPDATE ON owners FOR EACH ROW
    WHEN NEW.update_at = OLD.update_at    --- avoid infinite loop
BEGIN
  UPDATE owners SET update_at=CURRENT_TIMESTAMP WHERE folder=NEW.folder AND owner=NEW.owner;
END;

CREATE TABLE messages (
  id          INT          NOT NULL,
  folder      VARCHAR(25)  REFERENCES folders(name) ON DELETE CASCADE ON UPDATE CASCADE,
  author      VARCHAR(25)  REFERENCES users(login) ON UPDATE CASCADE,
  subject     VARCHAR(53)  NOT NULL,
  message     TEXT         NOT NULL,
  permanent   INT          DEFAULT 0 NOT NULL,
  shutdown    INT          DEFAULT 0 NOT NULL,
  expiration  TIMESTAMP    NOT NULL,
  create_at   TIMESTAMP    DEFAULT CURRENT_TIMESTAMP NOT NULL,
  update_at   TIMESTAMP    DEFAULT CURRENT_TIMESTAMP NOT NULL,
  PRIMARY KEY (id, folder)
) WITHOUT ROWID;
CREATE INDEX messages_idx_shutdown ON messages(shutdown);
CREATE INDEX messages_idx_expiration ON messages(expiration);

CREATE TABLE seen (
  login       VARCHAR(25) REFERENCES users(login) ON DELETE CASCADE ON UPDATE CASCADE,
  folder      VARCHAR(25) REFERENCES folders(name) ON DELETE CASCADE ON UPDATE CASCADE,
  msgid       INT,
  PRIMARY KEY (folder, login, msgid),
  CONSTRAINT read_fk_id_folder
    FOREIGN KEY (msgid, folder)
    REFERENCES messages(id, folder)
    ON DELETE CASCADE
    ON UPDATE CASCADE
) WITHOUT ROWID;

CREATE TABLE mark (
  login       VARCHAR(25) REFERENCES users(login) ON DELETE CASCADE ON UPDATE CASCADE,
  folder      VARCHAR(25) REFERENCES folders(name) ON DELETE CASCADE ON UPDATE CASCADE,
  msgid       INT,
  PRIMARY KEY (folder, login, msgid),
  CONSTRAINT mark_fk_id_folder
    FOREIGN KEY (msgid, folder)
    REFERENCES messages(id, folder)
    ON DELETE CASCADE
    ON UPDATE CASCADE
) WITHOUT ROWID;

CREATE TABLE access (
  login       VARCHAR(25) REFERENCES users(login) ON DELETE CASCADE ON UPDATE CASCADE,
  folder      VARCHAR(25) REFERENCES folders(name) ON DELETE CASCADE ON UPDATE CASCADE,
  PRIMARY KEY (login, folder),
) WITHOUT ROWID;

--- TODO: The following is incomplete:
--- User configs.
CREATE TABLE config (
  login       VARCHAR(25) REFERENCES users(login) ON DELETE CASCADE ON UPDATE CASCADE,
  folder      VARCHAR(25) REFERENCES folders(name) ON DELETE CASCADE ON UPDATE CASCADE,
  always      INT     NOT NULL DEFAULT 0,
  alert       INT     NOT NULL DEFAULT 0,  --- 0=no, 1=brief, 2=readnew
  always      INT     NOT NULL
) WITHOUT ROWID;

--- System configs.
CREATE TABLE system (
  default_expire  INT     NOT NULL DEFAULT -1,
  expire_limit    INT     NOT NULL DEFAULT -1,
) WITHOUT ROWID;
