// Package users manages users.
package users

import (
	"errors"
	"fmt"
	"strings"

	"git.lyda.ie/kevin/bulletin/folders"
	"github.com/chzyer/readline"
	_ "modernc.org/sqlite" // Loads sqlite driver.
)

// UserData is the type for holding user data. Things like preferences,
// unread message counts, signatures, etc.
type UserData struct {
	Login          string
	FullName       string
	Folders        *folders.Store
	CurrentFolder  string
	CurrentMessage int
}

// User is the user for this process.  It is loaded by the `Verify` function.
var User *UserData

// ValidName makes sure that an account name is a valid name.
func ValidName(login string) error {
	if login == "" {
		return errors.New("empty account is invalid")
	}
	if strings.ContainsAny(login, "./") {
		return fmt.Errorf("account name '%s' is invalid", login)
	}
	return nil
}

// Open verifies that an account exists.
func Open(login, name string) error {
	err := ValidName(login)
	if err != nil {
		return err
	}
	User = &UserData{
		Login: login,
	}

	User.Folders, err = folders.Open(login)
	if err != nil {
		return err
	}
	user, _ := User.Folders.GetUser(login)

	if user.Login != login {
		user.Login = login
		user.Admin = 0

		user.Name = name
		if name == "" {
			fmt.Printf("Welcome new user %s\n", login)
			user.Name, err = GetLine("please enter your name: ")
			if err != nil {
				return err
			}
		}

		err = User.Folders.AddUser(*user)
		if err != nil {
			return err
		}
		fmt.Println("User successfully created. Enjoy!")
	}
	if user.Disabled == 1 {
		return errors.New("User is disabled")
	}
	User.FullName = user.Name
	User.CurrentFolder = "GENERAL"
	// TODO: get the most recent unread message.
	// User.CurrentMessage =

	return nil
}

// Close closes the resources open for the account.
func (u *UserData) Close() {
	u.Folders.Close()
}

// GetLine gets a line.
func GetLine(prompt string) (string, error) {
	rl, err := readline.New(prompt)
	if err != nil {
		return "", err
	}
	defer rl.Close()
	line, err := rl.Readline()
	return line, err
}
