// Package accounts manages accounts.
package accounts

import (
	"database/sql"
	"errors"
	"fmt"
	"os"
	"path"
	"strings"

	"github.com/adrg/xdg"
	_ "modernc.org/sqlite" // Loads sqlite driver.
)

// UserData is the type for holding user data. Things like preferences,
// unread message counts, signatures, etc.
type UserData struct {
	Account  string
	FullName string
	pref     *sql.DB
	bull     *sql.DB
}

// User is the user for this process.  It is loaded by the `Verify` function.
var User *UserData

// ValidName makes sure that an account name is a valid name.
func ValidName(acc string) error {
	if acc == "" {
		return errors.New("empty account is invalid")
	}
	if strings.ContainsAny(acc, "./") {
		return fmt.Errorf("account name '%s' is invalid", acc)
	}
	return nil
}

// Open verifies that an account exists.
func Open(acc string) error {
	err := ValidName(acc)
	if err != nil {
		return err
	}
	User = &UserData{
		Account: acc,
	}

	prefdir := path.Join(xdg.ConfigHome, "BULLETIN")
	err = os.MkdirAll(prefdir, 0700)
	if err != nil {
		return errors.New("account preference directory problem")
	}
	User.pref, err = sql.Open("sqlite", path.Join(prefdir, fmt.Sprintf("%s.%s", acc, ".db")))
	if err != nil {
		return errors.New("account preference database problem")
	}
	// TODO: run prefs migration - move this to a storage module.

	bulldir := path.Join(xdg.DataHome, "BULLETIN")
	err = os.MkdirAll(bulldir, 0700)
	if err != nil {
		return errors.New("bulletin directory problem")
	}
	User.bull, err = sql.Open("sqlite", path.Join(bulldir, "bboard.db"))
	if err != nil {
		return errors.New("bulletin database problem")
	}
	// TODO: run prefs migration - move this to a storage module.

	return nil
}

// Close closes the resources open for the account.
func (u *UserData) Close() {
	u.pref.Close()
	u.bull.Close()
}

// IsAdmin returns true if the user is an admin
func IsAdmin(acc string) bool {
	if acc == "admin" {
		return true
	}
	// TODO: Look up account otherwise.
	return false
}
