// Package dclish A DCL-like command line parser.
package dclish

import (
	"fmt"
	"strings"
)

// ActionFunc is the function that a command runs.
type ActionFunc func(*Command) error

// Flag is a flag for a command.
type Flag struct {
	Value       string
	Default     string
	Description string
	// TODO: Toggle bool
}

// Flags is the list of flags.
type Flags map[string]*Flag

// Command contains the definition of a command, it's flags and subcommands.
type Command struct {
	Flags       Flags
	Args        []string
	MaxArgs     int
	MinArgs     int
	Commands    []*Command
	Action      ActionFunc
	Description string
}

// Commands is the full list of commands.
type Commands map[string]*Command

// ParseAndRun parses a command line and runs the command.
func (c Commands) ParseAndRun(line string) error {
	// TODO: this doesn't handle a DCL command line completely.
	words := strings.Fields(line)
	cmd, ok := c[strings.ToUpper(words[0])]
	if !ok {
		wordup := strings.ToUpper(words[0])
		possibles := []string{}
		for word := range c {
			if strings.HasPrefix(word, wordup) {
				possibles = append(possibles, word)
			}
		}
		switch len(possibles) {
		case 0:
			fmt.Printf("ERROR: Unknown command '%s'\n", words[0])
			return nil
		case 1:
			cmd = c[possibles[0]]
		default:
			fmt.Printf("ERROR: Ambiguous command '%s' (matches %s)\n", words[0], possibles)
			return nil
		}
	}
	if cmd.Action == nil {
		fmt.Printf("ERROR: Command not implemented:\n%s\n", cmd.Description)
		return nil
	}
	for flg := range cmd.Flags {
		cmd.Flags[flg].Value = cmd.Flags[flg].Default
	}
	cmd.Args = []string{}
	if len(words) == 1 {
		return cmd.Action(cmd)
	}
	// TODO: need to clean this up.
	args := words[1:]
	for i := range args {
		if strings.HasPrefix(args[i], "/") {
			flag, val, assigned := strings.Cut(args[i], "=")
			if assigned {
				wordup := strings.ToUpper(flag)
				flg, ok := cmd.Flags[wordup]
				if !ok {
					fmt.Printf("ERROR: Flag '%s' not recognised.", args[i])
				}
				flg.Value = val
			} else {
				wordup := strings.ToUpper(args[i])
				value := "true"
				if strings.HasPrefix(wordup, "/NO") {
					wordup = strings.Replace(wordup, "/NO", "/", 1)
					value = "false"
				}
				flg, ok := cmd.Flags[wordup]
				if !ok {
					fmt.Printf("ERROR: Flag '%s' not recognised.", args[i])
				}
				flg.Value = value
			}
		} else {
			cmd.Args = append(cmd.Args, args[i])
		}
	}
	return cmd.Action(cmd)
}
