// Package repl implements the main event loop.
package repl

import (
	"fmt"
	"strings"

	"git.lyda.ie/kevin/bulletin/dclish"
)

var helpmap = map[string]string{
	"FOLDERS": `All messages are divided into separate folders.  The default folder is
GENERAL.  New folders can be created by any user.  As an example, the
following creates a folder for GAMES related messages: 
 
BULLETIN> CREATE GAMES
Enter a one line description of folder.
GAMES
 
To see the list of available folders, use DIRECTORY/FOLDERS.  To select
a specific folder, use the SELECT command.  
 
If a user selects a folder and enters the SET READNEW command, that
user will be alerted of topics of new messages at login time, and will 
then be given the option of reading them.  Similar to READNEW is SHOWNEW,
which displays the topics but doesn't prompt to read them.  Even less is
SET BRIEF, which will cause only a one line output indicating that there
are new messages in the folder.  There also is the SET NOTIFY option,
which will cause a message to be broadcast to a user's terminal alerting
the user that a new message has been added.  Any of these options can be
the default for the folder by using the /DEFAULT switch on the command.
 
A folder can be restricted to only certain users, if desired.  This is 
done by specifying CREATE/PRIVATE.  Afterwards, access to the folder is 
controlled by the creator by the SET [NO]ACCESS command.  If /SEMIPRIVATE
rather than /PRIVATE is specified, all users can read the messages in the
folder, but only those give access can add messages.
 
A folder can be converted into a remote folder using CREATE/NODE or SET
NODE.  A remote folder is one which points to a folder on a remote DECNET
node.  Messages added to a remote node are actually stored on the folder
on the remote node.  The BULLCP process (created by BULLETIN/STARTUP)
must be running on the remote node for this option to be used.

A folder can be specified as a SYSTEM folder, i.e. one in which SYSTEM/
SHUTDOWN/BROADCAST messages can be added.  By default, the GENERAL folder
is a SYSTEM folder (and cannot be changed).  One use for this is to create
a remote SYSTEM folder which is shared by all nodes, so that the GENERAL
folder is used for messages pertaining only to the local host, while the
remote folder is used for messages pertaining to all nodes.  Another
use is to create a folder for posting SYSTEM messages only meant for a
certain UIC group.  This is done by creating a PRIVATE SYSTEM folder, and
giving access to that UIC group.  Only users in that UIC group will see
the messages in that folder when they log in.`,
	"CTRL-C": `Except for when BULLETIN is awaiting input from the terminal, a 
CTRL-C will cause BULLETIN to abort the execution of any command.  If
BULLETIN is waiting for terminal input, a CTRL-C will cause BULLETIN
to return to the BULLETIN> prompt.  If for some reason the user wishes
to suspend BULLETIN, CTRL-Y will usually do so.  However, this is not
always true, as BULLETIN will ignore the CTRL-Y if it has a data file
opened at the time. (Otherwise it would be possible to put the files
in a state such that they would be inaccessible by other users.)
`,
}

func init() {
	for c := range commands {
		helpmap[c] = commands[c].Description
	}
}

// ActionHelp handles the `HELP` command.
func ActionHelp(cmd *dclish.Command) error {
	if len(cmd.Args) == 0 {
		fmt.Printf("%s\n", cmd.Description)
		return nil
	}
	wordup := strings.ToUpper(cmd.Args[0])
	helptext, ok := helpmap[wordup]
	if !ok {
		possibles := []string{}
		for word := range helpmap {
			if strings.HasPrefix(word, wordup) {
				possibles = append(possibles, word)
			}
		}
		switch len(possibles) {
		case 0:
			fmt.Printf("ERROR: Topic not found: '%s'.\n", cmd.Args[0])
			return nil
		case 1:
			helptext = helpmap[possibles[0]]
		default:
			fmt.Printf("ERROR: Ambiguous topic '%s' (matches %s)\n", cmd.Args[0], possibles)
			return nil
		}

	}
	fmt.Printf("%s\n\n", helptext)
	return nil
}
