package repl

import (
	"fmt"
	"strings"

	"git.lyda.ie/kevin/bulletin/ask"
	"git.lyda.ie/kevin/bulletin/dclish"
	"git.lyda.ie/kevin/bulletin/key"
	"git.lyda.ie/kevin/bulletin/storage"
	"git.lyda.ie/kevin/bulletin/this"
	"git.lyda.ie/kevin/bulletin/users"
)

// ActionUser handles the `USER` command.
func ActionUser(cmd *dclish.Command) error {
	fmt.Println(cmd.Description)
	fmt.Println(`
The following commands are available:

  ADD        ADMIN      DELETE     DISABLE    ENABLE     LIST
  MOD        NOADMIN    NOMOD`)
	fmt.Println()
	return nil
}

// ActionUserAdd handles the `USER ADD` command.
func ActionUserAdd(cmd *dclish.Command) error {
	login := strings.ToUpper(cmd.Args[0])
	err := users.ValidLogin(login)
	if err != nil {
		fmt.Printf("ERROR: %s.\n", err)
		return nil
	}
	return nil
}

// ActionUserList handles the `USER LIST` command.
func ActionUserList(_ *dclish.Command) error {
	if this.User.Admin == 0 {
		fmt.Println("ERROR: You are not an admin.")
		return nil
	}
	ctx := storage.Context()
	userlist, err := this.Q.ListUsers(ctx)
	if err != nil {
		fmt.Printf("ERROR: Failed to list users (%s).\n", err)
		return nil
	}
	// TODO: nicer output for user.
	for _, u := range userlist {
		fmt.Printf("%s\n", u)
	}
	return nil
}

// ActionUserDelete handles the `USER DELETE` command.
func ActionUserDelete(cmd *dclish.Command) error {
	if this.User.Admin == 0 {
		fmt.Println("ERROR: You are not an admin.")
		return nil
	}
	u, err := users.ValidExistingLogin(this.Q, cmd.Args[0])
	if err != nil || u.Login == "" {
		fmt.Println("ERROR: User not found.")
		return nil
	}
	ctx := storage.Context()
	err = this.Q.DeleteUser(ctx, u.Login)
	if err != nil {
		fmt.Printf("ERROR: Failed to delete user (%s).\n", err)
		return nil
	}
	fmt.Println("User deleted.")
	return nil
}

func actionUserEnable(cmd *dclish.Command, disabled int64, doing string) error {
	if this.User.Admin == 0 {
		fmt.Println("ERROR: You are not an admin.")
		return nil
	}
	u, err := users.ValidExistingLogin(this.Q, cmd.Args[0])
	if err != nil || u.Login == "" {
		fmt.Println("ERROR: User not found.")
		return nil
	}
	if u.Disabled == disabled {
		fmt.Printf("User already %sd.\n", doing)
		return nil
	}
	ctx := storage.Context()
	err = this.Q.UpdateUserDisabled(ctx, storage.UpdateUserDisabledParams{
		Login:    u.Login,
		Disabled: disabled,
	})
	if err != nil {
		fmt.Printf("ERROR: Failed to %s user (%s).\n", doing, err)
		return nil
	}
	fmt.Printf("User %sd.\n", doing)
	return nil
}

// ActionUserEnable handles the `USER ENABLE` command.
func ActionUserEnable(cmd *dclish.Command) error {
	return actionUserEnable(cmd, 0, "enable")
}

// ActionUserDisable handles the `USER DISABLE` command.
func ActionUserDisable(cmd *dclish.Command) error {
	return actionUserEnable(cmd, 1, "disable")
}

func actionUserAdmin(cmd *dclish.Command, admin int64, doing string) error {
	if this.User.Admin == 0 {
		fmt.Println("ERROR: You are not an admin.")
		return nil
	}
	u, err := users.ValidExistingLogin(this.Q, cmd.Args[0])
	if err != nil || u.Login == "" {
		fmt.Println("ERROR: User not found.")
		return nil
	}
	if u.Admin == admin {
		fmt.Printf("User is already %s.\n", doing)
		return nil
	}
	ctx := storage.Context()
	err = this.Q.UpdateUserAdmin(ctx, storage.UpdateUserAdminParams{
		Login: u.Login,
		Admin: admin,
	})
	if err != nil {
		fmt.Printf("ERROR: Failed to make user %s (%s).\n", doing, err)
		return nil
	}
	fmt.Printf("User is now %s.\n", doing)
	return nil
}

// ActionUserAdmin handles the `USER ADMIN` command.
func ActionUserAdmin(cmd *dclish.Command) error {
	return actionUserAdmin(cmd, 1, "an admin")
}

// ActionUserNoadmin handles the `USER NOADMIN` command.
func ActionUserNoadmin(cmd *dclish.Command) error {
	return actionUserAdmin(cmd, 0, "not an admin")
}

func actionUserMod(cmd *dclish.Command, mod int64, doing string) error {
	if this.User.Admin == 0 {
		fmt.Println("ERROR: You are not an admin.")
		return nil
	}
	u, err := users.ValidExistingLogin(this.Q, cmd.Args[0])
	if err != nil || u.Login == "" {
		fmt.Println("ERROR: User not found.")
		return nil
	}
	if u.Moderator == mod {
		fmt.Printf("User is already %s.\n", doing)
		return nil
	}
	ctx := storage.Context()
	err = this.Q.UpdateUserMod(ctx, storage.UpdateUserModParams{
		Login:     u.Login,
		Moderator: mod,
	})
	if err != nil {
		fmt.Printf("ERROR: Failed to make user %s (%s).\n", doing, err)
		return nil
	}
	fmt.Printf("User is now %s.\n", doing)
	return nil
}

// ActionUserMod handles the `USER MOD` command.
func ActionUserMod(cmd *dclish.Command) error {
	return actionUserMod(cmd, 1, "a moderator")
}

// ActionUserNomod handles the `USER NOMOD` command.
func ActionUserNomod(cmd *dclish.Command) error {
	return actionUserMod(cmd, 0, "not a moderator")
}

// ActionSSH handles the `SSH` command.
func ActionSSH(cmd *dclish.Command) error {
	fmt.Println(cmd.Description)
	fmt.Println(`
The following commands are available:

  ADD        DELETE     LIST`)
	fmt.Println()
	return nil
}

// ActionSSHAdd handles the `SSH ADD` command.
func ActionSSHAdd(cmd *dclish.Command) error {
	if this.User.Admin == 0 && len(cmd.Args) == 1 {
		fmt.Println("ERROR: You are not an admin.")
		return nil
	}
	login := this.User.Login
	if len(cmd.Args) == 1 {
		login = cmd.Args[0]
	}
	u, err := users.ValidExistingLogin(this.Q, login)
	if err != nil || u.Login == "" {
		fmt.Println("ERROR: User not found.")
		return nil
	}
	sshkey, err := ask.GetLine("Enter ssh public key: ")
	if err != nil {
		fmt.Printf("ERROR: Failed to read ssh key (%s).\n", err)
		return nil
	}
	key.Add(login, sshkey)
	fmt.Println("Key is added.")
	return nil
}

// ActionSSHList handles the `SSH LIST` command.
func ActionSSHList(cmd *dclish.Command) error {
	if this.User.Admin == 0 && len(cmd.Args) == 1 {
		fmt.Println("ERROR: You are not an admin.")
		return nil
	}
	login := this.User.Login
	if len(cmd.Args) == 1 {
		login = cmd.Args[0]
	}
	u, err := users.ValidExistingLogin(this.Q, login)
	if err != nil || u.Login == "" {
		fmt.Println("ERROR: User not found.")
		return nil
	}
	keys, err := key.List(login)
	if err != nil {
		fmt.Printf("ERROR: Problem listing keys (%s).\n", err)
		return nil
	}
	fmt.Printf("The %d keys:\n  %s\n", len(keys), strings.Join(keys, "\n  "))
	return nil
}

// ActionSSHDelete handles the `SSH DELETE` command.
func ActionSSHDelete(cmd *dclish.Command) error {
	if this.User.Admin == 0 && len(cmd.Args) == 1 {
		fmt.Println("ERROR: You are not an admin.")
		return nil
	}
	login := this.User.Login
	if len(cmd.Args) == 1 {
		login = cmd.Args[0]
	}
	u, err := users.ValidExistingLogin(this.Q, login)
	if err != nil || u.Login == "" {
		fmt.Println("ERROR: User not found.")
		return nil
	}
	keys, err := key.List(login)
	if err != nil {
		fmt.Printf("ERROR: Problem listing keys (%s).\n", err)
		return nil
	}
	if len(keys) == 0 {
		fmt.Println("No keys to delete.")
		return nil
	}
	choice, err := ask.Choose("Choose a key to delete:", keys)
	if err != nil {
		fmt.Printf("ERROR: Problem choosing key (%s).\n", err)
		return nil
	}
	if choice < 0 {
		fmt.Println("Aborted.")
		return nil
	}
	err = key.Delete(keys[choice])
	if err != nil {
		fmt.Printf("ERROR: Problem deleting key (%s).\n", err)
		return nil
	}
	fmt.Println("Key deleted.")
	return nil
}
