// Package repl implements the main event loop.
package repl

import (
	"fmt"
	"strings"

	"github.com/carlmjohnson/versioninfo"

	"git.lyda.ie/kevin/bulletin/dclish"
	"git.lyda.ie/kevin/bulletin/folders"
	"git.lyda.ie/kevin/bulletin/storage"
	"git.lyda.ie/kevin/bulletin/this"
)

// ActionShow handles the `SHOW` command.
func ActionShow(cmd *dclish.Command) error {
	fmt.Println(cmd.Description)
	fmt.Println(`
The following commands are available:

  FLAGS       FOLDER      NEW         PRIVILEGES  USER        VERSION`)
	fmt.Println()
	return nil
}

// ActionShowFlags handles the `SHOW FLAGS` command.
func ActionShowFlags(_ *dclish.Command) error {
	flagset := false
	fmt.Printf("For the selected folder %s:\n", this.Folder.Name)
	if this.Folder.Notify != 0 {
		fmt.Println("  NOTIFY is set.")
		flagset = true
	}
	if this.Folder.Readnew != 0 {
		fmt.Println("  READNEW is set.")
		flagset = true
	}
	if this.Folder.Brief != 0 {
		fmt.Println("  BRIEF is set.")
		flagset = true
	}
	if this.Folder.Shownew != 0 {
		fmt.Println("  SHOWNEW is set.")
		flagset = true
	}
	if !flagset {
		fmt.Println("  No flags are set.")
	}
	return nil
}

// ActionShowFolder handles the `SHOW FOLDER` command.
func ActionShowFolder(cmd *dclish.Command) error {
	ctx := storage.Context()

	folder := this.Folder
	if len(cmd.Args) == 1 {
		folder = folders.FindFolder(cmd.Args[0])
	}
	if folder.Name == "" {
		fmt.Println("ERROR: Specified folder was not found.")
	}
	owners, err := this.Q.GetOwners(ctx, folder.Name)
	if err != nil || len(owners) == 0 {
		fmt.Printf("ERROR: This folder seems to lack owners (%s).\n", err)
		return nil
	}

	full := false
	if cmd.Flags["/FULL"].Value == "true" {
		// TODO: Check permissions.
		full = true
	}
	fmt.Printf("Settings for %s.\n", folder.Name)
	if full {
		switch folder.Visibility {
		case folders.FolderPublic:
			fmt.Println("  Folder is a public folder.")
		case folders.FolderSemiPrivate:
			fmt.Println("  Folder is a semi-private folder.")
		case folders.FolderPrivate:
			fmt.Println("  Folder is a private folder.")
		}
	}
	switch folder.Expire {
	case -1:
		fmt.Println("  Default expiration is permanent.")
	case 0:
		fmt.Println("  No default expiration set.")
	default:
		fmt.Printf("  Default expiration is %d days.\n", folder.Expire)
	}
	if folder.System != 0 {
		fmt.Println("  SYSTEM has been set.")
	}
	// TODO: Review SHOW_FOLDER in bulletin5.for.
	if folder.Always != 0 {
		fmt.Println("  ALWAYS has been set.")
	}
	if this.Folder.Notify != 0 {
		fmt.Println("  Default is NOTIFY.")
	}
	if this.Folder.Readnew != 0 {
		fmt.Println("  Default is READNEW.")
	}
	if this.Folder.Brief != 0 {
		fmt.Println("  Default is BRIEF.")
	}
	if this.Folder.Shownew != 0 {
		fmt.Println("  Default is SHOWNEW.")
	}

	return nil
}

// ActionShowNew handles the `SHOW NEW` command.
func ActionShowNew(_ *dclish.Command) error {
	fmt.Println("TODO: implement ActionShowNew.")
	return nil
}

// ActionShowPrivileges handles the `SHOW PRIVILEGES` command.
func ActionShowPrivileges(_ *dclish.Command) error {
	fmt.Println("TODO: implement ActionShowPrivileges.")
	return nil
}

// ActionShowUser handles the `SHOW USER` command.
func ActionShowUser(cmd *dclish.Command) error {
	// Parse the options.
	login := this.User.Login
	if len(cmd.Args) == 1 {
		if this.User.Admin == 0 {
			fmt.Println("ERROR: You are not an admin.")
			return nil
		}
		login = strings.ToUpper(cmd.Args[0])
	}
	if cmd.Flags["/ALL"].Set {
		if this.User.Admin == 0 {
			fmt.Println("ERROR: You are not an admin.")
			return nil
		}
		if len(cmd.Args) == 1 {
			fmt.Println("ERROR: You can't specify a user with /ALL.")
			return nil
		}
	}
	var showLogin int64
	if cmd.Flags["/LOGIN"].Value == "true" {
		if this.User.Admin == 0 {
			fmt.Println("ERROR: You are not an admin.")
			return nil
		}
		showLogin = 1
	}
	folder := this.Folder
	if cmd.Flags["/FOLDER"].Set {
		fmt.Printf("%v\n", cmd.Flags["/FOLDER"].Value)
		if cmd.Flags["/FOLDER"].Value != "" {
			folder = folders.FindFolder(cmd.Flags["/FOLDER"].Value)
			if folder.Name == "" {
				fmt.Println("ERROR: Folder does not exist.")
				return nil
			}
		}
	}
	/*
		TODO: need to add this.
		var since time.Time
		var err error
		if cmd.Flags["/SINCE"].Value != "" {
			if cmd.Flags["/LOGIN"].Set || cmd.Flags["/FOLDER"].Set {
				fmt.Println("ERROR: Must set /[NO]LOGIN or /FOLDER.")
				return nil
			}
			since, err = ParseDate(cmd.Flags["/SINCE"].Value)
			if err != nil {
				fmt.Println("ERROR: Invalid date specified.")
				return nil
			}
		}
	*/

	// Actually do the thing.
	ctx := storage.Context()
	if cmd.Flags["/LOGIN"].Set && cmd.Flags["/FOLDER"].Set {
		rows, err := this.Q.GetLastReadByEnabled(ctx,
			storage.GetLastReadByEnabledParams{
				Folder:   folder.Name,
				Disabled: showLogin,
			})
		if err != nil {
			fmt.Printf("ERROR: Failed to get list (%s).\n", err)
			return nil
		}
		fmt.Println("User          Folder                     Message #")
		for _, r := range rows {
			fmt.Printf("%-12s  %-25s  % 4d\n", r.Author, folder.Name, r.ID)
		}
	} else if cmd.Flags["/ALL"].Set && cmd.Flags["/FOLDER"].Set {
		rows, err := this.Q.GetLastRead(ctx, folder.Name)
		if err != nil {
			fmt.Printf("ERROR: Failed to get list (%s).\n", err)
			return nil
		}
		fmt.Println("User          Folder                     Message #")
		for _, r := range rows {
			fmt.Printf("%-12s  %-25s  % 4d\n", r.Author, folder.Name, r.ID)
		}
	} else if cmd.Flags["/FOLDER"].Set {
		r, err := this.Q.GetLastReadByUser(ctx,
			storage.GetLastReadByUserParams{
				Folder: folder.Name,
				Author: login,
			})
		if err != nil {
			fmt.Printf("ERROR: Failed to get list (%s).\n", err)
			return nil
		}
		fmt.Println("User          Folder                     Message #")
		fmt.Printf("%-12s  %-25s  % 4d\n", r.Author, folder.Name, r.ID)
	} else if cmd.Flags["/LOGIN"].Set {
		rows, err := this.Q.GetLastLoginByEnabled(ctx, showLogin)
		if err != nil {
			fmt.Printf("ERROR: Failed to get list (%s).\n", err)
			return nil
		}
		fmt.Println("User          Last Login")
		for _, r := range rows {
			fmt.Printf("%-12s  %s\n", r.Login, r.LastLogin.Format("2006-01-02 15:04:05"))
		}
	} else if cmd.Flags["/ALL"].Set {
		rows, err := this.Q.GetLastLogin(ctx)
		if err != nil {
			fmt.Printf("ERROR: Failed to get list (%s).\n", err)
			return nil
		}
		fmt.Println("User          Last Login")
		for _, r := range rows {
			fmt.Printf("%-12s  %s\n", r.Login, r.LastLogin.Format("2006-01-02 15:04:05"))
		}
	} else {
		r, err := this.Q.GetLastLoginByLogin(ctx, login)
		if err != nil {
			fmt.Printf("ERROR: Failed to get list (%s).\n", err)
			return nil
		}
		fmt.Println("User          Last Login")
		fmt.Printf("%-12s  %s\n", r.Login, r.LastLogin.Format("2006-01-02 15:04:05"))
	}
	return nil
}

// ActionShowVersion handles the `SHOW VERSION` command.
func ActionShowVersion(_ *dclish.Command) error {
	rev := versioninfo.Revision
	if len(rev) > 7 {
		rev = rev[len(rev)-7:]
	}
	fmt.Printf("BULLETIN Version 2.0.7g (%s)\n", rev)
	fmt.Printf("Linked on %s\n",
		versioninfo.LastCommit.Format("2006-05-04 15:02:01"))
	return nil
}
