// Package dclish A DCL-like command line parser.
package dclish

import (
	"fmt"
	"strings"
	"unicode"
)

// ActionFunc is the function that a command runs.
type ActionFunc func(*Command) error

// Flag is a flag for a command.
type Flag struct {
	OptArg      bool
	Value       string
	Default     string
	Description string
}

// Flags is the list of flags.
type Flags map[string]*Flag

// Command contains the definition of a command, it's flags and subcommands.
type Command struct {
	Flags       Flags
	Args        []string
	MaxArgs     int
	MinArgs     int
	Commands    []*Command
	Action      ActionFunc
	Description string
}

// Commands is the full list of commands.
type Commands map[string]*Command

func split(line string) []string {
	words := []string{}
	buf := strings.Builder{}
	state := "start"
	for _, c := range line {
		switch state {
		case "start":
			if unicode.IsSpace(c) {
				continue
			}
			if c == '"' {
				state = "dquote"
			} else if c == '\'' {
				state = "squote"
			} else {
				state = "raw"
				buf.WriteRune(c)
			}
		case "dquote":
			if c == '"' {
				words = append(words, buf.String())
				buf.Reset()
				state = "start"
			} else {
				buf.WriteRune(c)
			}
		case "squote":
			if c == '\'' {
				words = append(words, buf.String())
				buf.Reset()
				state = "start"
			} else {
				buf.WriteRune(c)
			}
		case "dquote-raw":
			if c == '"' {
				state = "raw"
			}
			buf.WriteRune(c)
		case "squote-raw":
			if c == '\'' {
				state = "raw"
			}
			buf.WriteRune(c)
		case "raw":
			if unicode.IsSpace(c) {
				words = append(words, buf.String())
				buf.Reset()
				state = "start"
			} else if c == '/' {
				words = append(words, buf.String())
				buf.Reset()
				state = "raw"
				buf.WriteRune(c)
			} else if c == '"' {
				state = "dquote-raw"
				buf.WriteRune(c)
			} else if c == '\'' {
				state = "squote-raw"
				buf.WriteRune(c)
			} else {
				buf.WriteRune(c)
			}
		}
	}
	if len(buf.String()) > 0 {
		words = append(words, buf.String())
	}
	return words
}

// ParseAndRun parses a command line and runs the command.
func (c Commands) ParseAndRun(line string) error {
	// TODO: this doesn't handle a DCL command line completely.
	words := split(line)
	cmd, ok := c[strings.ToUpper(words[0])]
	if !ok {
		wordup := strings.ToUpper(words[0])
		possibles := []string{}
		for word := range c {
			if strings.HasPrefix(word, wordup) {
				possibles = append(possibles, word)
			}
		}
		switch len(possibles) {
		case 0:
			fmt.Printf("ERROR: Unknown command '%s'\n", words[0])
			return nil
		case 1:
			cmd = c[possibles[0]]
		default:
			fmt.Printf("ERROR: Ambiguous command '%s' (matches %s)\n",
				words[0], strings.Join(possibles, ", "))
			return nil
		}
	}
	if cmd.Action == nil {
		fmt.Printf("ERROR: Command not implemented:\n%s\n", cmd.Description)
		return nil
	}
	for flg := range cmd.Flags {
		cmd.Flags[flg].Value = cmd.Flags[flg].Default
	}
	cmd.Args = []string{}
	if len(words) == 1 {
		if len(cmd.Args) < cmd.MinArgs {
			fmt.Println("ERROR: Not enough args.")
			return nil
		}
		return cmd.Action(cmd)
	}
	// TODO: need to clean this up.
	args := words[1:]
	for i := range args {
		if strings.HasPrefix(args[i], "/") {
			flag, val, assigned := strings.Cut(args[i], "=")
			var wordup string
			if assigned {
				wordup = strings.ToUpper(flag)
			} else {
				wordup = args[i]
			}
			toggleValue := "true"
			flg, ok := cmd.Flags[wordup]
			if !ok {
				wordup = strings.Replace(wordup, "/NO", "/", 1)
				flg, ok = cmd.Flags[wordup]
				if !ok {
					fmt.Printf("ERROR: Flag '%s' not recognised.\n", args[i])
					return nil
				}
				toggleValue = "false"
			}
			if !flg.OptArg && assigned {
				fmt.Printf("ERROR: Flag '%s' is a toggle.\n", args[i])
				return nil
			}
			if flg.OptArg {
				if assigned {
					flg.Value = strings.Trim(val, "\"'")
				}
			} else {
				flg.Value = toggleValue
			}
		} else {
			if len(cmd.Args) == cmd.MaxArgs {
				fmt.Printf("ERROR: Too many args at '%s'\n", args[i])
				return nil
			}
			cmd.Args = append(cmd.Args, args[i])
		}
	}
	if len(cmd.Args) < cmd.MinArgs {
		fmt.Println("ERROR: Not enough args.")
		return nil
	}
	return cmd.Action(cmd)
}
