// Package folders are all the routines and sql for managing folders.
package folders

import (
	"database/sql"
	"errors"
	"strings"

	"git.lyda.ie/kevin/bulletin/storage"
	"git.lyda.ie/kevin/bulletin/this"
)

// ValidFolder validates the folder name for this user.
func ValidFolder(folder string) (storage.Folder, error) {
	if !IsAlphaNum(folder) {
		return storage.Folder{}, errors.New("Folder can only have letters and numbers")
	}
	correct := FindFolder(folder)
	if correct.Name == "" {
		return storage.Folder{}, errors.New("Unable to select the folder")
	}
	if !IsFolderReadable(correct.Name, this.User.Login) {
		return storage.Folder{}, errors.New("Unable to select the folder")
	}
	return correct, nil
}

// Values for FolderVisibility.
const (
	FolderPublic      int64 = 0
	FolderSemiPrivate       = 1
	FolderPrivate           = 2
)

// CreateFolder creates a new folder.
func CreateFolder(options storage.CreateFolderParams) error {
	if !IsAlphaNum(options.Name) {
		return errors.New("Folder can only have letters and numbers")
	}
	options.Name = strings.ToUpper(options.Name)

	ctx := storage.Context()
	err := this.Q.CreateFolder(ctx, options)
	if err != nil {
		return err
	}
	return err
}

// ListFolder provides a list of folders that this.User has access to.
func ListFolder() ([]storage.ListFolderRow, error) {
	ctx := storage.Context()
	rows, err := this.Q.ListFolder(ctx)
	if err != nil {
		return []storage.ListFolderRow{}, err
	}
	return rows, nil
}

// FindFolder finds a folder based on the prefix.
func FindFolder(name string) storage.Folder {
	ctx := storage.Context()
	folder, _ := this.Q.FindFolderExact(ctx, name)
	if folder.Name != "" {
		return folder
	}
	folder, _ = this.Q.FindFolderPrefix(ctx, sql.NullString{
		String: name,
		Valid:  true,
	})
	return folder
}

// IsFolderReadable checks if a user can read messages from a folder.
func IsFolderReadable(name, login string) bool {
	ctx := storage.Context()
	admin, _ := this.Q.IsUserAdmin(ctx, login)
	if admin == 1 {
		return true
	}
	found, _ := this.Q.IsFolderReadable(ctx, name, login)
	return found == 1
}

// IsFolderWriteable checks if a user can write messages into a folder.
func IsFolderWriteable(name, login string) bool {
	ctx := storage.Context()
	admin, _ := this.Q.IsUserAdmin(ctx, login)
	if admin == 1 {
		return true
	}
	found, _ := this.Q.IsFolderWriteable(ctx, name, login)
	return found == 1
}

// IsFolderOwner checks if a user is a folder owner.
func IsFolderOwner(folder, login string) bool {
	ctx := storage.Context()
	admin, _ := this.Q.IsUserAdmin(ctx, login)
	if admin == 1 {
		return true
	}
	found, _ := this.Q.IsFolderOwner(ctx, folder, login)
	return found == 1
}

// DeleteFolder deletes a folder.
func DeleteFolder(name string) error {
	ctx := storage.Context()
	err := this.Q.DeleteFolder(ctx, name)
	if err != nil {
		return err
	}
	return nil
}
