/*
Package this has the current state of the running bulletin process.

Each bulletin process is run by a single user.  The `this` package
tracks the current user state - it has the user login, the current
folder, the current message id and other things.
*/
package this

import (
	"errors"
	"fmt"
	"strings"

	"git.lyda.ie/kevin/bulletin/ask"
	"git.lyda.ie/kevin/bulletin/storage"
	"git.lyda.ie/kevin/bulletin/users"
	"github.com/jmoiron/sqlx"
)

// User is the user for this session.
var User storage.User

// Store is the store for this session.
var Store *sqlx.DB

// Q is the storage.Queries for this session.
var Q *storage.Queries

// Folder is the current folder.
var Folder storage.Folder

// MsgID is the current message id.
var MsgID int64

// ReadFirstCall tracks whether this is the first time READ has been
// used in a folder since changing into it.
var ReadFirstCall bool

// System has the information about the system.
var System storage.System

// StartThis starts a session.
func StartThis(login string) error {
	// Validate the login name.
	login = strings.ToUpper(login)
	err := users.ValidLogin(login)
	if err != nil {
		return err
	}

	// Connect to the DB.
	Store, err = storage.Open()
	if err != nil {
		return err
	}
	Q = storage.New(Store.DB)

	ctx := storage.Context()
	System, err = Q.GetSystem(ctx)
	if err != nil {
		return err
	}
	fmt.Printf("Connected to the %s BULLETIN system.\n\n", System.Name)

	User, err = Q.GetUser(ctx, login)

	if User.Login != login {
		fmt.Printf("Welcome new user %s\n", login)
		name, err := ask.GetLine("please enter your name: ")
		if err != nil {
			return err
		}

		User, err = Q.AddUser(ctx, storage.AddUserParams{
			Login: login,
			Name:  name,
		})
		if err != nil {
			return err
		}
		fmt.Println("User successfully created. Enjoy!")
	} else {
		User.LastLogin, _ = Q.UpdateUserLastLogin(ctx, User.Login)
	}
	if User.Name == "" {
		fmt.Printf("Welcome new user %s\n", User.Login)
		name, err := ask.GetLine("please enter your name: ")
		if err != nil {
			return err
		}
		User.Name = name
		Q.UpdateUserName(ctx, User.Name, User.Login)
	}
	if User.Disabled == 1 {
		return errors.New("User is disabled")
	}

	Folder, err = Q.GetFolder(ctx, "GENERAL")
	if err != nil {
		return err
	}

	ReadFirstCall = true
	MsgID, err = Q.NextMsgid(ctx, Folder.Name, 0, User.Login)
	if MsgID == 0 {
		MsgID, err = Q.NextMsgidIgnoringSeen(ctx, Folder.Name, 0)
	}

	ShowAlerts(true)

	return nil
}
