package folders

import (
	"errors"
	"time"

	"git.lyda.ie/kevin/bulletin/storage"
	"git.lyda.ie/kevin/bulletin/this"
)

// CreateMessage creates a new folder.
func CreateMessage(author, subject, message, folder string, permanent, shutdown int, expiration *time.Time) error {
	ctx := storage.Context()
	if expiration == nil {
		days, err := this.Q.GetFolderExpire(ctx, folder)
		if err != nil {
			return err
		}
		if days <= 0 {
			// TODO: Get from site config.
			days = 14
		}
		exp := time.Now().AddDate(0, 0, int(days))
		expiration = &exp
	}
	// TODO: replace _ with rows and check.
	err := this.Q.CreateMessage(ctx, storage.CreateMessageParams{
		Folder:     folder,
		Author:     author,
		Subject:    subject,
		Message:    message,
		Permanent:  int64(permanent),
		Shutdown:   int64(shutdown),
		Expiration: *expiration,
	})
	// TODO: process this error a bit more to give a better error message.
	return err
}

// ReadMessage reads a message for a user.
func ReadMessage(login, folder string, msgid int64) (*storage.Message, error) {
	ctx := storage.Context()
	msg, err := this.Q.ReadMessage(ctx, storage.ReadMessageParams{
		Folder: folder,
		ID:     msgid,
	})
	if err != nil {
		return nil, err
	}

	if msg.ID != int64(msgid) || msgid == 0 {
		return nil, errors.New("Specified message was not found")
	}
	err = this.Q.SetMessageSeen(ctx, storage.SetMessageSeenParams{
		Login:  login,
		Folder: folder,
		Msgid:  int64(msgid),
	})

	return &msg, nil
}

// MarkSeen marks a list of messages as seen.
func MarkSeen(msgids []int64) error {
	ctx := storage.Context()

	for _, msgid := range msgids {
		this.Q.SetMessageSeen(ctx, storage.SetMessageSeenParams{
			Login:  this.User.Login,
			Folder: this.Folder.Name,
			Msgid:  msgid,
		})
	}
	return nil
}

// MarkUnseen marks a list of messages as unseen.
func MarkUnseen(msgids []int64) error {
	ctx := storage.Context()

	for _, msgid := range msgids {
		this.Q.UnsetMessageSeen(ctx, storage.UnsetMessageSeenParams{
			Login:  this.User.Login,
			Folder: this.Folder.Name,
			Msgid:  msgid,
		})
	}
	return nil
}

// NextMsgid gets the next message id.
func NextMsgid(login, folder string, msgid int64) int64 {
	ctx := storage.Context()
	newid, err := this.Q.NextMsgid(ctx, storage.NextMsgidParams{
		Folder: folder,
		Login:  login,
		ID:     msgid,
	})
	if err != nil {
		return 0
	}
	return newid
}

// PrevMsgid gets the next message id.
func PrevMsgid(login, folder string, msgid int64) int64 {
	ctx := storage.Context()
	newid, err := this.Q.PrevMsgid(ctx, storage.PrevMsgidParams{
		Folder: folder,
		Login:  login,
		ID:     msgid,
	})
	if err != nil {
		return 0
	}
	return newid
}

// FirstMessage gets the first message in a folder.
func FirstMessage(folder string) int64 {
	ctx := storage.Context()
	first, err := this.Q.NextMsgidIgnoringSeen(ctx,
		storage.NextMsgidIgnoringSeenParams{
			Folder: folder,
			ID:     0,
		})
	if err != nil {
		return 0
	}
	return first
}

// ListMessages lists messages.
func ListMessages(folder string) ([]storage.Message, error) {
	ctx := storage.Context()
	// TODO: options aren't implemented - need to set them?
	rows, err := this.Q.ListMessages(ctx, folder)
	return rows, err
}

// DeleteMessages deletes a list of messages.
func DeleteMessages(msgids []int64) error {
	ctx := storage.Context()

	for _, msgid := range msgids {
		this.Q.DeleteMessage(ctx, storage.DeleteMessageParams{
			Folder: this.Folder.Name,
			ID:     msgid,
		})
	}
	return nil
}

// DeleteAllMessages deletes all messages in a folder.
func DeleteAllMessages() error {
	ctx := storage.Context()

	this.Q.DeleteAllMessages(ctx, this.Folder.Name)
	return nil
}
