package storage

import (
	"context"
	"embed"
	"errors"
	"os"
	"path"

	// Included to connect to sqlite.
	_ "github.com/golang-migrate/migrate/v4/database/sqlite"
	_ "modernc.org/sqlite"

	"github.com/adrg/xdg"
	"github.com/golang-migrate/migrate/v4"
	"github.com/golang-migrate/migrate/v4/source/iofs"
	"github.com/jmoiron/sqlx"
)

//go:embed migrations/*.sql
var migrationsFS embed.FS

// Open opens the bulletin database.
func Open() (*sqlx.DB, error) {
	// Determine path names and create components.
	bulldir := path.Join(xdg.DataHome, "BULLETIN")
	err := os.MkdirAll(bulldir, 0700)
	if err != nil {
		return nil, errors.New("bulletin directory problem")
	}
	dbfile := path.Join(bulldir, "bulletin.db")

	// Run db migrations if needed.
	migrations, err := iofs.New(migrationsFS, "migrations")
	if err != nil {
		return nil, err
	}
	m, err := migrate.NewWithSourceInstance("iofs", migrations,
		"sqlite://"+dbfile+"?_pragma=foreign_keys(1)")
	if err != nil {
		return nil, err
	}
	defer m.Close()
	err = m.Up()
	if err != nil && err != migrate.ErrNoChange {
		return nil, err
	}

	// Connect to the db.
	db, err := sqlx.Connect("sqlite", "file://"+dbfile+"?_pragma=foreign_keys(1)")
	if err != nil {
		return nil, errors.New("bulletin database problem")
	}

	// Prepare to watch for schema skew - see version.go.
	InitialiseSkewChecker(db, m)

	return db, nil
}

// Context the standard context for storage calls.
func Context() context.Context {
	return context.TODO()
}
