// Package key manages the authorized keys file.
package key

import (
	"bufio"
	"bytes"
	"errors"
	"fmt"
	"io"
	"net/http"
	"os"
	"path"
	"strings"

	"github.com/adrg/xdg"
	"golang.org/x/crypto/ssh"
	"golang.org/x/sys/unix"
)

var keytemplate = `command="%s -u %s",no-port-forwarding,no-X11-forwarding,no-agent-forwarding %s`

// Add adds an ssh key to the `authorized_keys` file.
func Add(login, public string) error {
	// Parse and verify the key.
	public = strings.TrimSpace(public)
	theKey, _, _, _, err := ssh.ParseAuthorizedKey([]byte(public))
	if err != nil {
		return err
	}

	// Find the bulletin binary.
	bulletin, err := os.Executable()
	if err != nil {
		return err
	}

	// File system management.
	sshdir := path.Join(xdg.Home, ".ssh")
	err = os.MkdirAll(sshdir, 0700)
	if err != nil {
		return err
	}
	keyfile := path.Join(sshdir, "authorized_keys")

	// Open and lock the authorized_keys file.
	f, err := os.OpenFile(keyfile, os.O_RDWR|os.O_CREATE, 0600)
	if err != nil {
		return err
	}
	defer f.Close()
	if err := unix.Flock(int(f.Fd()), unix.LOCK_EX); err != nil {
		return err
	}
	defer unix.Flock(int(f.Fd()), unix.LOCK_UN) // unlock after we're done

	// Check for duplicates.
	keycontent, err := io.ReadAll(f)
	if err != nil {
		return err
	}
	if bytes.Contains(keycontent, ssh.MarshalAuthorizedKey(theKey)) {
		return errors.New("key already exists")
	}

	// Generate and write the key.
	keyline := fmt.Sprintf(keytemplate, bulletin, login, string(ssh.MarshalAuthorizedKey(theKey)))
	n, err := f.WriteString(keyline)
	if err != nil {
		return err
	}
	if n != len(keyline) {
		return fmt.Errorf("Failed to write authorized_key fully %d of %d chars written",
			n, len(keyline))
	}

	return nil
}

// List returns a list of ssh keys for this user.
func List(login string) ([]string, error) {
	keys := []string{}

	// Find the bulletin binary.
	bulletin, err := os.Executable()
	if err != nil {
		return keys, err
	}

	// File system management.
	sshdir := path.Join(xdg.Home, ".ssh")
	err = os.MkdirAll(sshdir, 0700)
	if err != nil {
		return keys, err
	}
	keyfile := path.Join(sshdir, "authorized_keys")

	// Open the authorized_keys file.
	f, err := os.OpenFile(keyfile, os.O_RDWR|os.O_CREATE, 0600)
	if err != nil {
		return keys, err
	}
	defer f.Close()

	// look for lines.
	scanner := bufio.NewScanner(f)
	for scanner.Scan() {
		keyline := bytes.TrimSpace(scanner.Bytes())
		if len(keyline) == 0 {
			continue
		}
		public, _, options, _, err := ssh.ParseAuthorizedKey([]byte(keyline))
		if err != nil {
			return keys, err
		}
		for i := range options {
			opts := strings.SplitN(options[i], "=", 2)
			if len(opts) != 2 || opts[0] != "command" {
				continue
			}
			cmd := strings.Split(strings.Trim(opts[1], "\" "), " ")
			if len(cmd) != 3 {
				return keys, fmt.Errorf("Unexpected command in authorized keys file (%s)", opts[1])
			}
			if cmd[0] != bulletin {
				return keys, fmt.Errorf("Unexpected bulletin in authorized keys file (%s)", opts[1])
			}
			if cmd[1] != "-u" {
				return keys, fmt.Errorf("Unexpected flag in authorized keys file (%s)", opts[1])
			}
			if cmd[2] == login {
				keys = append(keys, strings.Trim(string(ssh.MarshalAuthorizedKey(public)), "\n"))
			}
			break
		}
	}

	return keys, nil
}

// Delete removes the key.
func Delete(public string) error {
	keys := []string{}

	// Parse and verify the key.
	public = strings.TrimSpace(public)
	doomedRaw, _, _, _, err := ssh.ParseAuthorizedKey([]byte(public))
	if err != nil {
		return err
	}

	// File system management.
	sshdir := path.Join(xdg.Home, ".ssh")
	err = os.MkdirAll(sshdir, 0700)
	if err != nil {
		return err
	}
	keyfile := path.Join(sshdir, "authorized_keys")

	// Open the authorized_keys file.
	f, err := os.OpenFile(keyfile, os.O_RDWR|os.O_CREATE, 0600)
	if err != nil {
		return err
	}
	defer f.Close()
	if err := unix.Flock(int(f.Fd()), unix.LOCK_EX); err != nil {
		return err
	}
	defer unix.Flock(int(f.Fd()), unix.LOCK_UN) // unlock after we're done

	// look for lines.
	doomed := doomedRaw.Marshal()
	scanner := bufio.NewScanner(f)
	for scanner.Scan() {
		keyline := bytes.TrimSpace(scanner.Bytes())
		if len(keyline) == 0 {
			continue
		}
		potential, _, _, _, err := ssh.ParseAuthorizedKey(keyline)
		if err != nil {
			return err
		}
		if bytes.Compare(potential.Marshal(), doomed) != 0 {
			keys = append(keys, string(keyline))
		}
	}
	if _, err := f.Seek(0, os.SEEK_SET); err != nil {
		return fmt.Errorf("seek: %w", err)
	}
	f.Truncate(0)
	f.WriteString(strings.Join(keys, "\n") + "\n")

	return nil
}

// Fetch fetches keys and adds them.
func Fetch(login, nickname, username string) string {
	sites := map[string]string{
		"codeberg": "https://codeberg.org/%s.keys",
		"gitlab":   "https://gitlab.com/%s.keys",
		"github":   "https://github.com/%s.keys",
	}
	site := sites[strings.ToLower(nickname)]
	if site == "" {
		return fmt.Sprintln("ERROR: site nickname unknown.")
	}
	url := fmt.Sprintf(site, username)
	resp, err := http.Get(url)
	if err != nil {
		return fmt.Sprintf("ERROR: Failed to fetch ssh keys (%s).\n", err)
	}
	scanner := bufio.NewScanner(resp.Body)
	keys := 0
	for scanner.Scan() {
		keyline := string(bytes.TrimSpace(scanner.Bytes()))
		Add(login, keyline)
		keys++
	}
	switch keys {
	case 0:
		return fmt.Sprintln("No keys added.")
	case 1:
		return fmt.Sprintln("Key is added.")
	default:
		return fmt.Sprintf("%d keys added.\n", keys)
	}
}
