// Package dclish A DCL-like command line parser.
package dclish

import (
	"fmt"
	"strings"
)

// ActionFunc is the function that a command runs.
type ActionFunc func(*Command) error

// Flag is a flag for a command.
type Flag struct {
	Value       string
	Default     string
	Description string
}

// Flags is the list of flags.
type Flags map[string]*Flag

// Command contains the definition of a command, it's flags and subcommands.
type Command struct {
	Flags       Flags
	Args        []string
	Commands    []*Command
	Action      ActionFunc
	Description string
}

// Commands is the full list of commands.
type Commands map[string]*Command

// ParseAndRun parses a command line and runs the command.
func (c Commands) ParseAndRun(line string) error {
	// TODO: this doesn't handle a DCL command line completely.
	words := strings.Fields(line)
	fmt.Printf("TODO ParseAndRun need to parse flags: %s\n", words)
	cmd, ok := c[strings.ToUpper(words[0])]
	if !ok {
		fmt.Printf("ERROR: Unknown command '%s'\n", words[0])
		return nil
	}
	if cmd.Action == nil {
		fmt.Printf("ERROR: Command not implemented:\n%s\n", cmd.Description)
		return nil
	}
	for flg := range cmd.Flags {
		cmd.Flags[flg].Value = cmd.Flags[flg].Default
	}
	cmd.Args = []string{}
	if len(words) == 1 {
		return cmd.Action(cmd)
	}
	for i := range words[1:] {
		if strings.HasPrefix(words[i], "/") {
			flag, val, assigned := strings.Cut(words[i], "=")
			if assigned {
				wordup := strings.ToUpper(flag)
				flg, ok := cmd.Flags[wordup]
				if !ok {
					fmt.Printf("ERROR: Flag '%s' not recognised.", words[i])
				}
				flg.Value = val
			} else {
				// TODO: handle toggle flag.
				wordup := strings.ToUpper(words[i])
				value := "true"
				if strings.HasPrefix(wordup, "/NO") {
					wordup = strings.Replace(wordup, "/NO", "/", 1)
					value = "false"
				}
				flg, ok := cmd.Flags[wordup]
				if !ok {
					fmt.Printf("ERROR: Flag '%s' not recognised.", words[i])
				}
				flg.Value = value
			}
		}
	}
	return cmd.Action(cmd)
}
