// Package folders are all the routines and sql for managing folders.
package folders

import (
	"errors"
	"fmt"
	"strings"

	"github.com/jmoiron/sqlx"
)

// FolderVisibility is the folder visibility level.
type FolderVisibility string

// Values for FolderVisibility.
const (
	FolderPublic      FolderVisibility = "public"
	FolderSemiPrivate                  = "semi-private"
	FolderPrivate                      = "private"
)

// FolderCreateOptions are a list of folder options.
type FolderCreateOptions struct {
	Always      int
	Brief       int
	Description string
	Notify      int
	Owner       string
	Readnew     int
	Shownew     int
	System      int
	Expire      int
	Visibility  FolderVisibility
}

// CreateFolder creates a new folder.
func (s *Store) CreateFolder(name string, options FolderCreateOptions) error {
	if !IsAlphaNum(name) {
		return errors.New("Folder can only have letters and numbers")
	}
	name = strings.ToUpper(name)
	_, err := s.db.Exec(
		`INSERT INTO folders
			(name, always, brief, description, notify, owner, readnew,
			 shownew, system, expire, visibility)
			VALUES
			($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11)`,
		name,
		options.Always,
		options.Brief,
		options.Description,
		options.Notify,
		options.Owner,
		options.Readnew,
		options.Shownew,
		options.System,
		options.Expire,
		options.Visibility,
	)
	// TODO: turn _ into rows and make sure it was added.
	// TODO: process this error a bit more to give a better error message.
	return err
}

// FolderListOptions are a list of folder options.
type FolderListOptions struct {
}

// FolderListRow are a list of folder options.
type FolderListRow struct {
	Name        string `db:"name"`
	Count       int    `db:"count"`
	Description string `db:"description"`
}

// ListFolder creates a new folder.
func (s *Store) ListFolder(user string, _ FolderListOptions) ([]FolderListRow, error) {
	// TODO: get message counts.
	var rows *sqlx.Rows
	var err error
	if s.IsUserAdmin(user) {
		rows, err = s.db.Queryx(
			`SELECT name, 0 as count, description FROM folders
			ORDER BY name`,
		)
	} else {
		// TODO: limit access.
		rows, err = s.db.Queryx(
			`SELECT f.name, count(m.id) as count, f.description FROM folders AS f
			LEFT JOIN messages AS m ON f.name = m.folder
			GROUP By f.name
			ORDER BY f.name`)
	}
	flr := []FolderListRow{}
	if err != nil {
		// TODO: process this error a bit more to give a better error message.
		return flr, err
	}
	for rows.Next() {
		row := FolderListRow{}
		err := rows.StructScan(&row)
		if err != nil {
			// TODO: process this error a bit more to give a better error message.
			return flr, err
		}
		flr = append(flr, row)
	}
	return flr, nil
}

// FindFolder finds a folder based on the prefix.
func (s *Store) FindFolder(name string) string {
	var folder string
	s.db.Get(&folder, "SELECT name FROM folders where name = $1", name)
	if folder != "" {
		return folder
	}
	s.db.Get(&folder,
		`SELECT name FROM folders where name LIKE $1
	     ORDER BY name LIMIT 1`, name+"%")
	return folder
}

// IsFolderAccess checks if a user can access a folder.
func (s *Store) IsFolderAccess(name, user string) bool {
	found := 0
	s.db.Get(&found,
		`SELECT 1 FROM folders AS f LEFT JOIN co_owners AS c ON f.name = c.folder
		   WHERE f.name = $1 AND
			       (f.visibility = "public"
						  OR (f.owner = $2 OR c.OWNER = $2))`,
		name, user)
	return found == 1
}

// IsFolderOwner checks if a user is a folder owner.
func (s *Store) IsFolderOwner(name, user string) bool {
	found := 0
	s.db.Get(&found,
		`SELECT 1 FROM folders AS f LEFT JOIN co_owners AS c
	   ON f.name = c.folder
		 WHERE f.name = $1 AND (f.owner = '$2' OR c.OWNER = '$2')`,
		name, user)
	return found == 1
}

// DeleteFolder creates a new folder.
func (s *Store) DeleteFolder(name string) error {
	// TODO: make sure user can delete this table.
	results, err := s.db.Exec("DELETE FROM folders WHERE name=$1", name)
	// TODO: process this error a bit more to give a better error message.
	if err != nil {
		return err
	}
	rows, err := results.RowsAffected()
	if err != nil {
		return err
	}
	if rows == 0 {
		return errors.New("No such folder found")
	}
	if rows != 1 {
		return fmt.Errorf("Unexpected number (%d) of folders removed", rows)
	}
	return nil
}
