// Package repl implements the main event loop.
package repl

import (
	"errors"
	"fmt"
	"strconv"
	"strings"

	"git.lyda.ie/kevin/bulletin/accounts"
	"git.lyda.ie/kevin/bulletin/dclish"
	"git.lyda.ie/kevin/bulletin/folders"
)

// ActionIndex handles the `INDEX` command.  This lists all the folders.
func ActionIndex(_ *dclish.Command) error {
	options := folders.FolderListOptions{}
	rows, err := accounts.User.Folders.ListFolder(accounts.User.Login, options)
	if err != nil {
		return err
	}
	fmt.Println("The following folders are present")
	fmt.Println("Name                       Count Description")
	for _, row := range rows {
		fmt.Printf("%-25s  %5d %s\n", row.Name, row.Count, row.Description)
	}
	return nil
}

// ActionCreate handles the `CREATE` command.  This creates a folder.
func ActionCreate(cmd *dclish.Command) error {
	// Populate options...
	options := folders.FolderCreateOptions{}
	if cmd.Flags["/ALWAYS"].Value == "true" {
		options.Always = 1
	}
	if cmd.Flags["/BRIEF"].Value == "true" {
		options.Brief = 1
	}
	if cmd.Flags["/DESCRIPTION"].Value != "" {
		options.Description = cmd.Flags["/DESCRIPTION"].Value
	}
	if cmd.Flags["/NOTIFY"].Value == "true" {
		options.Notify = 1
	}
	if cmd.Flags["/OWNER"].Value != "" {
		options.Owner = cmd.Flags["/OWNER"].Value
	} else {
		options.Owner = accounts.User.Login
	}
	if cmd.Flags["/READNEW"].Value == "true" {
		options.Readnew = 1
	}
	if cmd.Flags["/SHOWNEW"].Value == "true" {
		options.Shownew = 1
	}
	if cmd.Flags["/SYSTEM"].Value == "true" {
		options.System = 1
	}
	if cmd.Flags["/EXPIRE"].Value != "" {
		expire, err := strconv.Atoi(cmd.Flags["/EXPIRE"].Value)
		if err != nil {
			return fmt.Errorf("Invalid expiry value '%s'", cmd.Flags["/EXPIRE"].Value)
		}
		options.Expire = expire
	}
	options.Visibility = folders.FolderPublic
	if cmd.Flags["/PRIVATE"].Value == "true" && cmd.Flags["/SEMIPRIVATE"].Value == "true" {
		return errors.New("Private or semi-private - pick one")
	}
	if cmd.Flags["/PRIVATE"].Value == "true" {
		options.Visibility = folders.FolderPrivate
	}
	if cmd.Flags["/SEMIPRIVATE"].Value == "true" {
		options.Visibility = folders.FolderSemiPrivate
	}

	// Verify options...
	if options.Description == "" {
		var err error
		options.Description, err = accounts.GetLine("Enter one line description of folder: ")
		if err != nil {
			return nil
		}
	}
	if options.Description == "" || len(options.Description) > 53 {
		return errors.New("Description must exist and be under 53 characters")
	}
	err := accounts.User.Folders.CreateFolder(cmd.Args[0], options)
	// TODO: handle the /ID flag.
	return err
}

// ActionSelect handles the `SELECT` command.  This selects a folder.
func ActionSelect(cmd *dclish.Command) error {
	if strings.Contains(cmd.Args[0], "%") {
		return errors.New("Folder name cannot contain a %")
	}
	folder := accounts.User.Folders.FindFolder(cmd.Args[0])
	if folder == "" {
		return errors.New("Unable to select the folder")
	}
	if accounts.User.Folders.IsFolderAccess(folder, accounts.User.Login) {
		accounts.User.CurrentFolder = folder
		fmt.Printf("Folder has been set to '%s'.\n", folder)
		return nil
	}
	// TODO: Should be:
	//       WRITE(6,'('' You are not allowed to access folder.'')')
	//       WRITE(6,'('' See '',A,'' if you wish to access folder.'')')
	return errors.New("Unable to select the folder")
}

// ActionModify handles the `MODIFY` command.  This modifies a folder.
func ActionModify(cmd *dclish.Command) error {
	fmt.Printf("TODO: implement MODIFY:\n%s\n\n", cmd.Description)
	return nil
}

// ActionRemove handles the `REMOVE` command.  This modifies a folder.
func ActionRemove(cmd *dclish.Command) error {
	err := accounts.User.Folders.DeleteFolder(cmd.Args[0])
	if err == nil {
		fmt.Println("Folder removed.")
	}
	return err
}
