CREATE TABLE users (
  login       VARCHAR(25)  NOT NULL PRIMARY KEY,
  name        VARCHAR(53)  NOT NULL,
  admin       INT          DEFAULT 0,
  create_at   TIMESTAMP    DEFAULT CURRENT_TIMESTAMP NOT NULL,
  update_at   TIMESTAMP    DEFAULT CURRENT_TIMESTAMP NOT NULL
) WITHOUT ROWID;

CREATE TRIGGER users_after_update_update_at
  AFTER UPDATE ON users FOR EACH ROW
  WHEN NEW.update_at = OLD.update_at    --- avoid infinite loop
BEGIN
  UPDATE users SET update_at=CURRENT_TIMESTAMP WHERE login=NEW.login;
END;

INSERT INTO users (login, name, admin)
       VALUES ('SYSTEM', 'System User', 1);

CREATE TRIGGER users_before_update_protect
  AFTER UPDATE ON users FOR EACH ROW
  WHEN OLD.login = 'SYSTEM' AND (NEW.login != OLD.login OR NEW.admin != 1)
BEGIN
  SELECT RAISE (ABORT, 'SYSTEM user is protected');
END;

CREATE TRIGGER users_before_delete_protect
  BEFORE DELETE on users FOR EACH ROW
  WHEN OLD.login = 'SYSTEM'
BEGIN
  SELECT RAISE (ABORT, 'SYSTEM user is protected');
END;

CREATE TABLE folders (
  name        VARCHAR(25)  NOT NULL PRIMARY KEY,
  always      INT          DEFAULT 0,
  brief       INT          DEFAULT 0,
  description VARCHAR(53)  NOT NULL,
  notify      INT          DEFAULT 0,
  owner       VARCHAR(25)  REFERENCES users(login) ON UPDATE CASCADE,
  readnew     INT          DEFAULT 0,
  shownew     INT          DEFAULT 0,
  system      INT          DEFAULT 0,
  expire      INT          DEFAULT 14,
  visibility  TEXT         DEFAULT 'public',
  create_at   TIMESTAMP    DEFAULT CURRENT_TIMESTAMP NOT NULL,
  update_at   TIMESTAMP    DEFAULT CURRENT_TIMESTAMP NOT NULL
) WITHOUT ROWID;

CREATE TRIGGER folders_before_insert_validate
  BEFORE INSERT on folders
BEGIN
  SELECT
    CASE
      WHEN NEW.name != UPPER(NEW.name) OR NEW.name GLOB '*[^A-Z0-9_-]*' THEN
        RAISE (ABORT, 'Invalid folder name')
    END;
END;

CREATE TRIGGER folders_before_update_validate
  BEFORE UPDATE on folders
BEGIN
  SELECT
    CASE
      WHEN NEW.name != UPPER(NEW.name) OR NEW.name GLOB '*[^A-Z0-9_-]*' THEN
        RAISE (ABORT, 'Invalid folder name')
      WHEN OLD.name = 'GENERAL' AND OLD.name != NEW.name THEN
        RAISE (ABORT, 'GENERAL folder is protected')
    END;
END;

CREATE TRIGGER folders_after_update_update_at
  AFTER UPDATE ON folders FOR EACH ROW
    WHEN NEW.update_at = OLD.update_at    --- avoid infinite loop
BEGIN
  UPDATE folders SET update_at=CURRENT_TIMESTAMP WHERE name=NEW.name;
END;

CREATE TRIGGER folders_before_delete_protect
  BEFORE DELETE on folders FOR EACH ROW
  WHEN OLD.name = 'GENERAL'
BEGIN
  SELECT RAISE (ABORT, 'GENERAL folder is protected');
END;

INSERT INTO folders (name, description, system, shownew, owner)
       VALUES ('GENERAL', 'Default general bulletin folder.', 1, 1, 'SYSTEM');

CREATE TABLE co_owners (
  folder      VARCHAR(25)  REFERENCES folders(name) ON DELETE CASCADE ON UPDATE CASCADE,
  owner       VARCHAR(25)  REFERENCES users(login) ON UPDATE CASCADE,
  create_at   TIMESTAMP    DEFAULT CURRENT_TIMESTAMP NOT NULL,
  update_at   TIMESTAMP    DEFAULT CURRENT_TIMESTAMP NOT NULL,
  PRIMARY KEY (folder, owner)
) WITHOUT ROWID;

CREATE TRIGGER co_owners_after_update_update_at
  AFTER UPDATE ON co_owners FOR EACH ROW
    WHEN NEW.update_at = OLD.update_at    --- avoid infinite loop
BEGIN
  UPDATE co_owners SET update_at=CURRENT_TIMESTAMP WHERE folder=NEW.folder AND owner=NEW.owner;
END;
