/*
Package ask provides routines to ask questions of users. It handles
getting a line of text, getting a choice from a list and other things.

Underneath it uses the [github.com/chzyer/readline] module.
*/
package ask

import (
	"errors"
	"fmt"
	"os"
	"strconv"

	"github.com/chzyer/readline"
)

// CheckErr prints an error message and exits.
func CheckErr(err error) {
	if err != nil {
		fmt.Printf("ERROR: %s\n", err)
		os.Exit(1)
	}
}

// GetLine gets a line.
func GetLine(prompt string) (string, error) {
	rl, err := readline.New(prompt)
	if err != nil {
		return "", err
	}
	defer rl.Close()
	line, err := rl.Readline()
	return line, err
}

// GetInt64 gets a line.
func GetInt64(prompt string) (int64, error) {
	line, err := GetLine(prompt)
	if err != nil {
		return 0, err
	}
	return strconv.ParseInt(line, 10, 64)
}

// Choose presents a list and asks a user to choose one.
func Choose(prompt string, choices []string) (int, error) {
	fmt.Println(prompt)
	for i := range choices {
		fmt.Printf("  %d. %s\n", i+1, choices[i])
	}
	response, err := GetLine("Choose a number or enter q to abort: ")
	if err != nil {
		return -1, err
	}
	choice, err := strconv.Atoi(response)
	if err != nil {
		return -1, nil
	}
	choice--
	if choice < 0 || choice >= len(choices) {
		return -1, errors.New("Choice out of bounds")
	}
	return choice, nil
}

// YesNo converts an int64 to "yes" (`!= 0`) or "no" (`== 0`).
func YesNo(num int64) string {
	if num != 0 {
		return "yes"
	}
	return "no"
}
