// Package batch contains both interactive and non-interactive maintenence
// commands.
//
// The interactive batch commands are as follows:
//
//   - `install` does the initial install for bulletin.  This will create
//     and seed the database and create the initial user.  It will also
//     create the user's crontab entries.
//   - `new-user` creates a new user.
//
// The non-interactive batch commands are run from the user's crontab.
//
//   - `reboot` is supposed to be run on system reboot and will delete
//     all shutdown messages.
//   - `expire` removes all expired messages and deletes broadcast messages
//     older than 3 days.
package batch

import (
	_ "embed"
	"errors"
	"fmt"
	"os"
	"path"
	"strings"
	"text/template"
	"time"

	"git.lyda.ie/pp/bulletin/ask"
	"git.lyda.ie/pp/bulletin/key"
	"git.lyda.ie/pp/bulletin/storage"
	"git.lyda.ie/pp/bulletin/users"
	"github.com/adrg/xdg"
)

//go:embed crontab
var crontabTemplate string

// Reboot deletes all messages with `shutdown` set.
func Reboot() int {
	ctx := storage.Context()

	store, err := storage.Open()
	ask.CheckErr(err)
	q := storage.New(store.DB)

	rows, err := q.DeleteAllShutdownMessages(ctx)
	ask.CheckErr(err)
	fmt.Printf("Removed %d shutdown messages\n", rows)
	return 0
}

// Expire deletes all messages that have hit their expiration date.
func Expire() int {
	ctx := storage.Context()

	store, err := storage.Open()
	ask.CheckErr(err)
	q := storage.New(store.DB)

	rows, err := q.DeleteAllExpiredMessages(ctx)
	ask.CheckErr(err)
	_, err = store.ExecContext(ctx, "VACUUM")
	ask.CheckErr(err)
	fmt.Printf("Expired %d messages\n", rows)
	return 0
}

// Install is an interactive command used to install the crontab.
func Install() int {
	ctx := storage.Context()

	// Check if install has run before.
	touchfile := path.Join(xdg.Home, ".bulletin-installed")
	if _, err := os.Stat(touchfile); err == nil {
		ask.CheckErr(errors.New("~/.bulletin-installed exists; BULLETIN install has run"))
	} else {
		if !errors.Is(err, os.ErrNotExist) {
			fmt.Println("ERROR: Unknown error checking in ~/.bulletin-installed exists.")
			fmt.Println("ERROR: Can't see if BULLETIN has been installed.")
			ask.CheckErr(err)
		}
	}

	// Connect to the database.
	store, err := storage.Open()
	ask.CheckErr(err)
	q := storage.New(store.DB)

	// Create the initial users.
	fmt.Println(`Welcome to the BULLETIN install process.  This will create
the following files:

  * ~/.local/share/BULLETIN/bulletin.db - BULLETIN's folder and user db.
	* ~/.bulletin-installed - A file to make sure install isn't run again.
	* ~/.config/BULLETIN/* - history files for each user.
	* ~/.ssh/authorized_keys - key lines to let users connect to BULLETIN

To complete the installation, please enter the login, name and ssh key for
the first user.`)
	system := storage.System{}
	system.Name, err = ask.GetLine("Enter system name: ")
	system.Name = strings.ToUpper(system.Name)
	ask.CheckErr(err)
	system.DefaultExpire, err = ask.GetInt64("Enter default expiry in days (180): ")
	if err != nil {
		system.DefaultExpire = 180
	}
	system.ExpireLimit, err = ask.GetInt64("Enter expiration limit in days (365): ")
	if err != nil {
		system.ExpireLimit = 365
	}
	err = q.SetSystem(ctx, system.Name, system.DefaultExpire, system.ExpireLimit)
	ask.CheckErr(err)

	login, err := ask.GetLine("Enter login of initial user: ")
	login = strings.ToUpper(login)
	ask.CheckErr(err)
	name, err := ask.GetLine("Enter name of initial user: ")
	ask.CheckErr(err)
	sshkey, err := ask.GetLine("Enter ssh public key of initial user: ")
	ask.CheckErr(err)

	// Seed data.
	seedMsgs, err := GetSeedMessages()
	ask.CheckErr(err)
	ask.CheckErr(q.SeedUserSystem(ctx))
	ask.CheckErr(q.SeedFolderGeneral(ctx))
	_, err = q.AddUser(ctx, storage.AddUserParams{
		Login: login,
		Name:  name,
	})
	key.Add(login, sshkey)
	userCreated := map[string]bool{
		"SYSTEM": true,
		login:    true,
	}
	for i := range seedMsgs {
		if !userCreated[seedMsgs[i].Login] {
			_, err = q.AddUser(ctx, storage.AddUserParams{
				Login:    seedMsgs[i].Login,
				Name:     seedMsgs[i].Name,
				Disabled: 1,
			})
			ask.CheckErr(err)
			userCreated[seedMsgs[i].Login] = true
		}
		ask.CheckErr(q.SeedCreateMessage(ctx, storage.SeedCreateMessageParams{
			Folder:     "GENERAL",
			Author:     seedMsgs[i].Login,
			Subject:    seedMsgs[i].Subject,
			Message:    seedMsgs[i].Message,
			CreateAt:   seedMsgs[i].Date,
			Expiration: time.Now().UTC(),
		}))
	}

	// Install crontab.
	bulletin, err := os.Executable()
	ask.CheckErr(err)
	crontab := &strings.Builder{}
	template.Must(template.New("crontab").Parse(crontabTemplate)).
		Execute(crontab, map[string]string{"Bulletin": bulletin})
	fmt.Printf("Adding this to crontab:\n\n%s\n", crontab.String())
	err = installCrontab(crontab.String())
	ask.CheckErr(err)

	// Mark that install has happened.
	err = touch(touchfile)
	ask.CheckErr(err)

	return 0
}

// NewUser creates a new user based on command line arguments.
func NewUser(args []string) int {
	// Make sure we have enough args.
	if len(args) != 3 {
		fmt.Println("ERROR: Must supply login, site nickname and site username.")
		return 1
	}

	// Create a user if missing.
	login := strings.ToUpper(args[0])
	err := users.ValidLogin(login)
	ask.CheckErr(err)
	store, err := storage.Open()
	ask.CheckErr(err)
	q := storage.New(store.DB)
	ctx := storage.Context()
	u, err := q.GetUser(ctx, login)
	if u.Login == "" {
		u, err = q.AddUser(ctx, storage.AddUserParams{
			Login: login,
		})
		ask.CheckErr(err)
	}
	if u.Login == "" {
		fmt.Println("ERROR: Failed to make user.")
		return 1
	}

	response := key.Fetch(u.Login, args[1], args[2])
	fmt.Println(response)
	if strings.HasPrefix(response, "ERROR") {
		return 1
	}
	return 0
}
