package batch

import (
	"embed"
	"fmt"
	"io/fs"
	"path"
	"strings"
	"time"
)

//go:embed seed/*
var messagesFS embed.FS

// SeedMessage is the struct version of the messages.
type SeedMessage struct {
	Login   string
	Name    string
	Date    time.Time
	Subject string
	Message string
}

// GetSeedMessages gets the embedded seed messages and returns them in
// a data structure.
func GetSeedMessages() ([]SeedMessage, error) {
	var messages []SeedMessage

	entries, err := fs.ReadDir(messagesFS, "seed")
	if err != nil {
		return nil, fmt.Errorf("reading directory: %w", err)
	}

	for _, entry := range entries {
		if entry.IsDir() {
			continue
		}

		data, err := fs.ReadFile(messagesFS, path.Join("seed", entry.Name()))
		if err != nil {
			return nil, fmt.Errorf("reading file %s: %w", entry.Name(), err)
		}

		lines := strings.Split(string(data), "\n")
		if len(lines) < 4 {
			continue // skip malformed
		}

		// Extract fields
		var login, name, subj string
		var date time.Time

		for i := 0; i < 3; i++ {
			line := strings.TrimSpace(lines[i])
			switch {
			case strings.HasPrefix(line, "From: "):
				rest := strings.TrimPrefix(line, "From: ")
				parts := strings.SplitN(rest, `"`, 2)
				if len(parts) == 2 {
					login = strings.TrimSpace(strings.TrimSuffix(parts[0], " "))
					name = strings.Trim(parts[1], `"`)
				} else {
					login = rest
				}
			case strings.HasPrefix(line, "Date: "):
				datestr := strings.TrimPrefix(line, "Date: ")
				// Expected format: 01-JAN-1982 12:00
				date, err = time.Parse("02-Jan-2006 15:04", datestr)
				if err != nil {
					return nil, fmt.Errorf("parsing date in %s: %w", entry.Name(), err)
				}
			case strings.HasPrefix(line, "Subj: "):
				subj = strings.TrimPrefix(line, "Subj: ")
			}
		}

		// Find start of message body
		bodyIndex := 3
		for bodyIndex < len(lines) && strings.TrimSpace(lines[bodyIndex]) != "" {
			bodyIndex++
		}
		body := strings.Join(lines[bodyIndex+1:], "\n")

		msg := SeedMessage{
			Login:   login,
			Name:    name,
			Date:    date,
			Subject: subj,
			Message: body,
		}
		messages = append(messages, msg)
	}

	return messages, nil
}
