/*
Package dclish A DCL-like command line parser.

From the OpenVMS manual on how to parse a DCL command line.

The following rules apply when entering DCL commands. Refer to
Chapter 5 for information about using extended file names in DCL
commands.

  - Use any combination of uppercase and lowercase letters. The DCL
    interpreter translates lowercase letters to uppercase. Uppercase and
    lowercase characters in parameter and qualifier values are equivalent
    unless enclosed in quotation marks (" ").
  - Separate the command name from the first parameter with at least one
    blank space or a tab.
  - Separate each additional parameter from the previous parameter or
    qualifier with at least one blank space or a tab.
  - Begin each qualifier with a slash (/). The slash serves as a separator
    and need not be preceded by blank spaces or tabs.
  - If a parameter or qualifier value includes a blank space or a tab,
    enclose the parameter or qualifier value in quotation marks.
  - You cannot specify null characters (<NUL>) on a DCL command line,
    even if you enclose the null character in quotation marks.
  - Include no more than 127 elements (parameters, qualifiers, and qualifier
    values) in each command line.
  - Each element in a command must not exceed 255 characters. The entire
    command must not exceed 1024 characters after all symbols1 and lexical
    functions2 are converted to their values.
  - You can abbreviate a command as long as the abbreviated name remains
    unique among the defined commands on a system. DCL looks only at the
    first four characters for uniqueness.
*/
package dclish

import (
	"errors"
	"fmt"
	"strings"
	"unicode"
)

// ActionFunc is the function that a command runs.
type ActionFunc func(*Command) error

// CompleterFunc is a function to provide completions for arguments for
// a given command.
type CompleterFunc func() []string

// Flag is a flag for a command. In the future setting a type would make
// things a little easier.
type Flag struct {
	OptArg      bool
	Value       string
	Default     string
	Description string
	Set         bool
}

// Flags is the list of flags.
type Flags map[string]*Flag

// Command contains the definition of a command, its flags, subcommands
// and a completer function for the arguments.  The number of args can
// be limited.
type Command struct {
	Flags       Flags
	Args        []string
	MaxArgs     int
	MinArgs     int
	Commands    Commands
	Action      ActionFunc
	Completer   CompleterFunc
	Description string
}

// Commands is the full list of commands.
type Commands map[string]*Command

func split(line string) []string {
	words := []string{}
	buf := strings.Builder{}
	state := "start"
	for _, c := range line {
		switch state {
		case "start":
			if unicode.IsSpace(c) {
				continue
			}
			if c == '"' {
				state = "dquote"
			} else if c == '\'' {
				state = "squote"
			} else {
				state = "raw"
				buf.WriteRune(c)
			}
		case "dquote":
			if c == '"' {
				words = append(words, buf.String())
				buf.Reset()
				state = "start"
			} else {
				buf.WriteRune(c)
			}
		case "squote":
			if c == '\'' {
				words = append(words, buf.String())
				buf.Reset()
				state = "start"
			} else {
				buf.WriteRune(c)
			}
		case "dquote-raw":
			if c == '"' {
				state = "raw"
			}
			buf.WriteRune(c)
		case "squote-raw":
			if c == '\'' {
				state = "raw"
			}
			buf.WriteRune(c)
		case "raw":
			if unicode.IsSpace(c) {
				words = append(words, buf.String())
				buf.Reset()
				state = "start"
			} else if c == '/' {
				words = append(words, buf.String())
				buf.Reset()
				state = "raw"
				buf.WriteRune(c)
			} else if c == '"' {
				state = "dquote-raw"
				buf.WriteRune(c)
			} else if c == '\'' {
				state = "squote-raw"
				buf.WriteRune(c)
			} else {
				buf.WriteRune(c)
			}
		}
	}
	if len(buf.String()) > 0 {
		words = append(words, buf.String())
	}
	return words
}

// PrefixMatch searches for a command in a list of possible commands.
func PrefixMatch(command string, commands []string) (string, error) {
	cmd := strings.ToUpper(command)
	possibles := []string{}
	for i := range commands {
		if strings.HasPrefix(cmd, commands[i]) {
			possibles = append(possibles, commands[i])
		}
	}
	switch len(possibles) {
	case 0:
		return "", fmt.Errorf("Unknown command '%s'", command)
	case 1:
		return possibles[0], nil
	default:
		return "", fmt.Errorf("Ambiguous command '%s' (matches %s)",
			command, strings.Join(possibles, ", "))
	}
}

// ParseAndRun parses a command line and runs the command.
func (c Commands) ParseAndRun(line string) error {
	// Split into words.
	words := split(line)

	return c.run(words)
}

func (c Commands) run(words []string) error {
	// Find the command.
	wordup := strings.ToUpper(words[0])
	cmd, ok := c[wordup]
	if !ok {
		possibles := []string{}
		for word := range c {
			if strings.HasPrefix(word, wordup) {
				possibles = append(possibles, word)
			}
		}
		switch len(possibles) {
		case 0:
			return fmt.Errorf("Unknown command '%s'", words[0])
		case 1:
			wordup = possibles[0]
			cmd = c[wordup]
		default:
			return fmt.Errorf("Ambiguous command '%s' (matches %s)",
				words[0], strings.Join(possibles, ", "))
		}
	}

	// Deal with subcommands.
	if len(cmd.Commands) > 0 {
		if len(words) == 1 {
			if cmd.Action == nil {
				return fmt.Errorf("missing subcommand for %s", wordup)
			}
			return cmd.Action(cmd)
		}
		return cmd.Commands.run(words[1:])
	}

	if cmd.Action == nil {
		return fmt.Errorf("Command not implemented:\n%s", cmd.Description)
	}
	for flg := range cmd.Flags {
		cmd.Flags[flg].Value = cmd.Flags[flg].Default
		cmd.Flags[flg].Set = false
	}
	cmd.Args = []string{}

	if len(words) == 1 {
		if len(cmd.Args) < cmd.MinArgs {
			return errors.New("Not enough args")
		}
		return cmd.Action(cmd)
	}
	args := words[1:]
	for i := range args {
		if strings.HasPrefix(args[i], "/") {
			flag, val, assigned := strings.Cut(args[i], "=")
			var wordup string
			if assigned {
				wordup = strings.ToUpper(flag)
			} else {
				wordup = strings.ToUpper(args[i])
			}
			toggleValue := "true"
			flg, ok := cmd.Flags[wordup]
			if !ok {
				wordup = strings.Replace(wordup, "/NO", "/", 1)
				flg, ok = cmd.Flags[wordup]
				if !ok {
					return fmt.Errorf("Flag '%s' not recognised", args[i])
				}
				toggleValue = "false"
			}
			if !flg.OptArg && assigned {
				return fmt.Errorf("Flag '%s' is a toggle", args[i])
			}
			if flg.Set {
				return fmt.Errorf("Flag '%s' is already set", args[i])
			}
			flg.Set = true
			if flg.OptArg {
				if assigned {
					flg.Value = strings.Trim(val, "\"'")
				}
			} else {
				flg.Value = toggleValue
			}
		} else {
			if len(cmd.Args) == cmd.MaxArgs {
				return fmt.Errorf("Too many args at '%s'", args[i])
			}
			cmd.Args = append(cmd.Args, args[i])
		}
	}
	if len(cmd.Args) < cmd.MinArgs {
		return errors.New("Not enough args")
	}
	return cmd.Action(cmd)
}
