/*
Package editor provides a simple editor.

This is implemented with [github.com/gdamore/tcell] and
[github.com/rivo/tview].  It would be good to make this work more like EDT.

In addition, it might be an idea to offer a set of editors that work
like EDT, nano, vi and emacs and allow users to set a preference.
*/
package editor

import (
	"fmt"

	"github.com/gdamore/tcell/v2"
	"github.com/rivo/tview"
)

// Editor is the editor for text files.
func Editor(placeholder, title, message string) (string, error) {
	theme := tview.Theme{
		PrimitiveBackgroundColor:    tcell.ColorDefault,
		ContrastBackgroundColor:     tcell.ColorDefault,
		MoreContrastBackgroundColor: tcell.ColorDefault,
		BorderColor:                 tcell.ColorDefault,
		TitleColor:                  tcell.ColorDefault,
		GraphicsColor:               tcell.ColorDefault,
		PrimaryTextColor:            tcell.ColorDefault,
		SecondaryTextColor:          tcell.ColorDefault,
		TertiaryTextColor:           tcell.ColorDefault,
		InverseTextColor:            tcell.ColorDefault,
		ContrastSecondaryTextColor:  tcell.ColorDefault,
	}
	tview.Styles = theme
	app := tview.NewApplication()

	textArea := tview.NewTextArea().
		SetPlaceholder(placeholder)
	textArea.SetTitle(title)
	helpInfo := tview.NewTextView().
		SetText(" Press F1 for help, press Ctrl-C to exit")
	position := tview.NewTextView().
		SetTextAlign(tview.AlignRight)
	pages := tview.NewPages()
	pages.SetBackgroundColor(tcell.ColorDefault)

	updateInfos := func() {
		fromRow, fromColumn, toRow, toColumn := textArea.GetCursor()
		if fromRow == toRow && fromColumn == toColumn {
			position.SetText(fmt.Sprintf("Row: %d, Column: %d ", fromRow, fromColumn))
		} else {
			position.SetText(fmt.Sprintf("From Row: %d, Column: %d - To Row: %d, To Column: %d ", fromRow, fromColumn, toRow, toColumn))
		}
	}

	textArea.SetMovedFunc(updateInfos)
	updateInfos()

	mainView := tview.NewGrid().
		SetRows(0, 1).
		AddItem(textArea, 0, 0, 1, 2, 0, 0, true).
		AddItem(helpInfo, 1, 0, 1, 1, 0, 0, false).
		AddItem(position, 1, 1, 1, 1, 0, 0, false)

	help1 := tview.NewTextView().
		SetText(`Navigation

Left arrow: Move left.
Right arrow: Move right.
Down arrow: Move down.
Up arrow: Move up.
Ctrl-A, Home: Move to the beginning of the current line.
Ctrl-E, End: Move to the end of the current line.
Ctrl-F, page down: Move down by one page.
Ctrl-B, page up: Move up by one page.
Alt-Up arrow: Scroll the page up.
Alt-Down arrow: Scroll the page down.
Alt-Left arrow: Scroll the page to the left.
Alt-Right arrow:  Scroll the page to the right.
Alt-B, Ctrl-Left arrow: Move back by one word.
Alt-F, Ctrl-Right arrow: Move forward by one word.

Press Enter for more help, press Escape to return.`)
	help2 := tview.NewTextView().
		SetText(`Editing

Type to enter text.
Ctrl-H, Backspace: Delete the left character.
Ctrl-D, Delete: Delete the right character.
Ctrl-K: Delete until the end of the line.
Ctrl-W: Delete the rest of the word.
Ctrl-U: Delete the current line.

Press Enter for more help, press Escape to return.`)
	help3 := tview.NewTextView().
		SetText(`Selecting Text

Move while holding Shift.
Double-click to select a word.

Clipboard

Ctrl-Q: Copy.
Ctrl-X: Cut.
Ctrl-V: Paste.

Undo

Ctrl-Z: Undo.
Ctrl-Y: Redo.

Press Enter for more help, press Escape to return.`)
	help := tview.NewFrame(help1).
		SetBorders(1, 1, 0, 0, 2, 2)
	help.SetBorder(true).
		SetTitle("Help").
		SetInputCapture(func(event *tcell.EventKey) *tcell.EventKey {
			if event.Key() == tcell.KeyEscape {
				pages.SwitchToPage("main")
				return nil
			} else if event.Key() == tcell.KeyEnter {
				switch {
				case help.GetPrimitive() == help1:
					help.SetPrimitive(help2)
				case help.GetPrimitive() == help2:
					help.SetPrimitive(help3)
				case help.GetPrimitive() == help3:
					help.SetPrimitive(help1)
				}
				return nil
			}
			return event
		})

	pages.AddAndSwitchToPage("main", mainView, true).
		AddPage("help", tview.NewGrid().
			SetColumns(0, 64, 0).
			SetRows(0, 22, 0).
			AddItem(help, 1, 1, 1, 1, 0, 0, true), true, false)

	app.SetInputCapture(func(event *tcell.EventKey) *tcell.EventKey {
		if event.Key() == tcell.KeyF1 {
			pages.ShowPage("help")
			return nil
		}
		return event
	})

	if message != "" {
		textArea.SetText(message, false)
	}
	err := app.SetRoot(pages, true).Run()
	if err != nil {
		return "", err
	}
	return textArea.GetText(), nil
}
