package folders

import (
	"errors"
	"time"

	"git.lyda.ie/pp/bulletin/storage"
	"git.lyda.ie/pp/bulletin/this"
)

// CreateMessage creates a new folder.
func CreateMessage(author, subject, message, folder string, permanent, shutdown int, expiration *time.Time) error {
	ctx := storage.Context()
	if expiration == nil {
		days, err := this.Q.GetFolderExpire(ctx, folder)
		if err != nil {
			return err
		}
		sysdef, err := this.Q.GetExpire(ctx)
		if days <= 0 {
			days = sysdef.DefaultExpire
		} else {
			days = min(days, sysdef.ExpireLimit)
		}
		exp := time.Now().AddDate(0, 0, int(days)).UTC()
		expiration = &exp
	}
	err := this.Q.CreateMessage(ctx, storage.CreateMessageParams{
		Folder:     folder,
		Author:     author,
		Subject:    subject,
		Message:    message,
		Permanent:  int64(permanent),
		Shutdown:   int64(shutdown),
		Expiration: *expiration,
	})
	return err
}

// GetMessage reads a message for a user.
func GetMessage(login, folder string, msgid int64) (*storage.Message, error) {
	ctx := storage.Context()
	msg, err := this.Q.ReadMessage(ctx, folder, msgid)
	if err != nil {
		return nil, err
	}

	if msg.ID != int64(msgid) || msgid == 0 {
		return nil, errors.New("Specified message was not found")
	}

	return &msg, nil
}

// MarkSeen marks a list of messages as seen.
func MarkSeen(msgids []int64) error {
	ctx := storage.Context()

	for _, msgid := range msgids {
		this.Q.SetMessageSeen(ctx, this.User.Login, this.Folder.Name, msgid)
	}
	return nil
}

// MarkUnseen marks a list of messages as unseen.
func MarkUnseen(msgids []int64) error {
	ctx := storage.Context()

	for _, msgid := range msgids {
		this.Q.UnsetMessageSeen(ctx, this.User.Login, this.Folder.Name, msgid)
	}
	return nil
}

// NextMsgid gets the next message id.
func NextMsgid(login, folder string, msgid int64) int64 {
	ctx := storage.Context()
	newid, err := this.Q.NextMsgid(ctx, folder, msgid, login)
	if err != nil {
		return 0
	}
	return newid
}

// PrevMsgid gets the next message id.
func PrevMsgid(login, folder string, msgid int64) int64 {
	ctx := storage.Context()
	newid, err := this.Q.PrevMsgid(ctx, folder, msgid, login)
	if err != nil {
		return 0
	}
	return newid
}

// FirstMessage gets the first message in a folder.
func FirstMessage(folder string) int64 {
	ctx := storage.Context()
	first, err := this.Q.NextMsgidIgnoringSeen(ctx, folder, 0)
	if err != nil {
		return 0
	}
	return first
}

// LastMessage gets the last message in a folder.
func LastMessage(folder string) int64 {
	ctx := storage.Context()
	last, err := this.Q.LastMsgidIgnoringSeen(ctx, folder)
	if err != nil {
		return 0
	}
	return last
}

// ListMessages lists messages.
func ListMessages(folder string) ([]storage.Message, error) {
	ctx := storage.Context()
	rows, err := this.Q.ListMessages(ctx, folder)
	return rows, err
}

// WroteAllMessages returns true if login wrote all msgids
func WroteAllMessages(login string, msgids []int64) bool {
	ctx := storage.Context()

	for _, msgid := range msgids {
		msg, err := this.Q.GetMessage(ctx, msgid, this.Folder.Name)
		if err != nil {
			return false
		}
		if msg.Author != login {
			return false
		}
	}
	return true
}

// DeleteMessages deletes a list of messages.
func DeleteMessages(msgids []int64) error {
	ctx := storage.Context()

	for _, msgid := range msgids {
		this.Q.DeleteMessage(ctx, msgid, this.Folder.Name)
	}
	return nil
}

// DeleteAllMessages deletes all messages in a folder.
func DeleteAllMessages() error {
	ctx := storage.Context()

	this.Q.DeleteAllMessages(ctx, this.Folder.Name)
	return nil
}

// SetMark sets a mark for message(s).
func SetMark(msgids []int64) error {
	ctx := storage.Context()

	for _, msgid := range msgids {
		this.Q.AddMark(ctx, this.Folder.Name, this.User.Login, msgid)
	}
	return nil
}

// UnsetMark sets a mark for message(s).
func UnsetMark(msgids []int64) error {
	ctx := storage.Context()

	for _, msgid := range msgids {
		this.Q.DeleteMark(ctx, this.Folder.Name, this.User.Login, msgid)
	}
	return nil
}
