// Package main is the base for bulletin.
//
// This is a Go implementation of the 1991 version of bulletin.
// The initial realease notes follow.  Note that at present there is no
// NEWS or MAIL support.
//
// You  are  about  to  receive  version  2.07  of the PFC BULLETIN.
//
// BULLETIN is public domain software.  (I will gladly accept
// recommendations for new features, not for changes that
// are due to "personal" preference.)
//
// As of V2.0, BULLETIN is able to read USENET NEWS
// via TCP/IP using either CMU, MULTINET, UCX, TWG, or
// via DECNET.  It can also serve as a NEWS gateway for
// DECNET nodes without direct access to the NEWS server,
// i.e. a DECNET node without Internet access will be able
// to read NEWS.
package main

import (
	"context"
	"fmt"
	"os"
	"strings"

	"git.lyda.ie/pp/bulletin/batch"
	"git.lyda.ie/pp/bulletin/repl"
	"git.lyda.ie/pp/bulletin/this"

	"github.com/urfave/cli/v3"
)

func main() {
	cmd := &cli.Command{
		Name:        "bulletin",
		Usage:       "a bulletin system",
		Description: "An interpretation of the VMS BULLETIN program.",
		Flags: []cli.Flag{
			&cli.StringFlag{
				Name:     "user",
				Aliases:  []string{"u"},
				Usage:    "user to run bulletin as",
				Required: true,
			},
			&cli.StringFlag{
				Name:    "batch",
				Aliases: []string{"b"},
				Usage:   "batch command",
			},
		},
		Action: func(_ context.Context, cmd *cli.Command) error {
			user := strings.ToUpper(cmd.String("user"))
			batchFlag := cmd.String("batch")

			if batchFlag != "" {
				if user != "SYSTEM" {
					fmt.Println("ERROR: can only run batch commands as SYSTEM.")
					os.Exit(1)
				}
				exitcode := 0
				switch batchFlag {
				case "reboot":
					exitcode = batch.Reboot()
				case "expire":
					exitcode = batch.Expire()
				case "install":
					exitcode = batch.Install()
				case "new-user":
					exitcode = batch.NewUser(cmd.Args().Slice())
				default:
					fmt.Println("ERROR: can only run batch commands as SYSTEM.")
					exitcode = 1
				}
				os.Exit(exitcode)
			}

			err := this.StartThis(user)
			if err != nil {
				return err
			}
			err = repl.Loop()
			if err != nil {
				return err
			}
			return nil
		},
	}

	err := cmd.Run(context.Background(), os.Args)
	if err != nil {
		fmt.Printf("ERROR: %s.\n", err)
	}
}
