/*
Package pager implements the pager.  This is used for any large block of
text that bulletin needs to show.

This is very unix specific. Using something like [github.com/gdamore/tcell]
would be more portable.
*/
package pager

import (
	"fmt"
	"os"
	"strings"

	"golang.org/x/sys/unix"
	"golang.org/x/term"
)

func getTerminalHeight() (int, error) {
	ws, err := unix.IoctlGetWinsize(0, unix.TIOCGWINSZ)
	if err != nil {
		return 0, err
	}
	return int(ws.Row), nil
}

// readKey reads a single key without needing Enter
func readKey() (byte, error) {
	var buf [1]byte
	_, err := os.Stdin.Read(buf[:])
	if err != nil {
		return 0, err
	}
	return buf[0], nil
}

// Pager pages through a string.
func Pager(content string) bool {
	lines := strings.Split(content, "\n")
	totalLines := len(lines)

	height, err := getTerminalHeight()
	if err != nil {
		fmt.Printf("%s\n", content)
		return true
	}
	pageSize := height - 1 // Reserve last line for prompt.

	// Set terminal raw mode.
	oldState, err := term.MakeRaw(int(os.Stdin.Fd()))
	if err != nil {
		fmt.Printf("%s\n", content)
		return true
	}
	defer term.Restore(int(os.Stdin.Fd()), oldState)

	start := 0
	for {
		// State for where we are.
		if start < 0 {
			start = 0
		}
		end := start + pageSize
		if end > totalLines {
			end = totalLines
		}

		// Clear prompt and display the chunk.
		fmt.Print("\r          \r")
		fmt.Print(strings.Join(lines[start:end], "\r\n"))

		if end == totalLines {
			fmt.Println("\r") // move to next line after paging ends
			return true
		}

		percent := (end * 100) / totalLines
		fmt.Printf("\r\nMore %d%%", percent)

		key, err := readKey()
		if err != nil {
			return false
		}

		switch key {
		case ' ': // page down
			start += pageSize
		case '\n', '\r': // line down
			start++
		case 'b': // page up
			start -= pageSize
		case 'q', 'Q': // quit
			fmt.Println("\r") // move cursor to next line
			return false
		}
	}
}
