package repl

import (
	"errors"
	"fmt"
	"strconv"
	"strings"

	"git.lyda.ie/pp/bulletin/ask"
	"git.lyda.ie/pp/bulletin/dclish"
	"git.lyda.ie/pp/bulletin/folders"
	"git.lyda.ie/pp/bulletin/storage"
	"git.lyda.ie/pp/bulletin/this"
)

// ActionIndex handles the `INDEX` command.  This lists all the folders.
//
// This originally existed as the subroutine FULL_DIR in bulletin9.for.
func ActionIndex(cmd *dclish.Command) error {

	if cmd.Flags["/MARKED"].Set {
		return errors.New("Not implemented yet")
	}
	if cmd.Flags["/NEW"].Set {
		return errors.New("Not implemented yet")
	}
	if cmd.Flags["/RESTART"].Set {
		return errors.New("Not implemented yet")
	}
	if cmd.Flags["/SEEN"].Set {
		return errors.New("Not implemented yet")
	}
	if cmd.Flags["/UNMARKED"].Set {
		return errors.New("Not implemented yet")
	}
	if cmd.Flags["/UNSEEN"].Set {
		return errors.New("Not implemented yet")
	}

	rows, err := folders.ListFolder()
	if err != nil {
		return err
	}
	fmt.Println("The following folders are present")
	fmt.Println("Name                       Count Description")
	for _, row := range rows {
		fmt.Printf("%-25s  %5d %s\n", row.Name, row.Count, row.Description)
	}
	return nil
}

// ActionCreate handles the `CREATE` command.  This creates a folder.
//
// This originally existed as the subroutine CREATE_FOLDER in bulletin4.for.
func ActionCreate(cmd *dclish.Command) error {
	// Populate options...
	options := storage.CreateFolderParams{}
	options.Name = cmd.Args[0]
	if cmd.Flags["/ALWAYS"].Value == "true" {
		options.Always = 1
	}
	if cmd.Flags["/BRIEF"].Value == "true" {
		options.Alert = 1
	}
	if cmd.Flags["/DESCRIPTION"].Value != "" {
		options.Description = cmd.Flags["/DESCRIPTION"].Value
	}
	if cmd.Flags["/READNEW"].Value == "true" {
		options.Alert = 2
	}
	if cmd.Flags["/SHOWNEW"].Value == "true" {
		options.Alert = 3
	}
	if cmd.Flags["/SYSTEM"].Value == "true" {
		options.System = 1
	}
	if cmd.Flags["/EXPIRE"].Value != "" {
		expire, err := strconv.Atoi(cmd.Flags["/EXPIRE"].Value)
		if err != nil {
			return fmt.Errorf("Invalid expiry value '%s'", cmd.Flags["/EXPIRE"].Value)
		}
		options.Expire = int64(expire)
	}
	if (cmd.Flags["/BRIEF"].Set && cmd.Flags["/READNEW"].Set) ||
		(cmd.Flags["/BRIEF"].Set && cmd.Flags["/SHOWNEW"].Set) ||
		(cmd.Flags["/READNEW"].Set && cmd.Flags["/SHOWNEW"].Set) {
		return errors.New("Can only set one of /BRIEF, /READNEW and /SHOWNEW")
	}
	options.Visibility = folders.FolderPublic
	if cmd.Flags["/PRIVATE"].Value == "true" && cmd.Flags["/SEMIPRIVATE"].Value == "true" {
		return errors.New("Private or semi-private - pick one")
	}
	if cmd.Flags["/PRIVATE"].Value == "true" {
		options.Visibility = folders.FolderPrivate
	}
	if cmd.Flags["/SEMIPRIVATE"].Value == "true" {
		options.Visibility = folders.FolderSemiPrivate
	}

	if cmd.Flags["/OWNER"].Value != "" {
		if this.User.Admin == 0 {
			return errors.New("Must be admin to specify /OWNER")
		}
		options.Owner = cmd.Flags["/OWNER"].Value
	} else {
		options.Owner = this.User.Login
	}

	// Verify options...
	if options.Description == "" {
		var err error
		options.Description, err = ask.GetLine("Enter one line description of folder: ")
		if err != nil {
			return nil
		}
	}
	if options.Description == "" || len(options.Description) > 53 {
		return errors.New("Description must exist and be under 53 characters")
	}
	err := folders.CreateFolder(options)
	return err
}

// ActionSelect handles the `SELECT` command.  This selects a folder.
//
// This is based on `SELECT_FOLDER` in bulletin5.for.
func ActionSelect(cmd *dclish.Command) error {
	if strings.Contains(cmd.Args[0], "%") {
		return errors.New("Folder name cannot contain a %")
	}
	folder := folders.FindFolder(cmd.Args[0])
	if folder.Name == "" {
		return errors.New("Unable to select the folder")
	}
	if folders.IsFolderReadable(folder.Name, this.User.Login) {
		this.Folder = folder
		this.ReadFirstCall = true
		fmt.Printf("Folder has been set to '%s'.\n", folder.Name)
		return nil
	}
	return errors.New("Unable to select the folder")
}

// ActionModify handles the `MODIFY` command.  This modifies a folder.
//
// This is based on `MODIFY_FOLDER` in bulletin1.for.
func ActionModify(cmd *dclish.Command) error {
	if this.User.Login != this.Folder.Owner && this.User.Admin == 0 {
		return errors.New("Must be folder owner or admin to modify the folder")
	}
	var err error
	description := this.Folder.Description
	if cmd.Flags["/DESCRIPTION"].Set {
		description, err = ask.GetLine("Enter one line description of folder: ")
		if err != nil {
			return err
		}
		if len(description) > 53 {
			return errors.New("Description must be < 53 characters")
		}
	}
	owner := this.Folder.Owner
	if cmd.Flags["/OWNER"].Set {
		if this.User.Admin == 0 {
			return errors.New("Must be an admin to modify the folder owner")
		}
		owner = cmd.Flags["/OWNER"].Value
	}
	name := this.Folder.Name
	if cmd.Flags["/NAME"].Set {
		name = cmd.Flags["/NAME"].Value
		if !folders.IsAlphaNum(name) {
			return errors.New("Invalid folder name")
		}
	}

	ctx := storage.Context()
	err = this.Q.UpdateFolderMain(ctx, storage.UpdateFolderMainParams{
		Description: description,
		Owner:       owner,
		NewName:     name,
		OldName:     this.Folder.Name,
	})
	return err
}

// ActionRemove handles the `REMOVE` command.  This modifies a folder.
//
// This originally existed as the subroutine REMOVE_FOLDER in bulletin5.for.
func ActionRemove(cmd *dclish.Command) error {
	if this.User.Login != this.Folder.Owner && this.User.Admin == 0 {
		return errors.New("Must be folder owner or admin to delete the folder")
	}
	if this.Folder.Name == "GENERAL" {
		return errors.New("Can't delete folder GENERAL")
	}
	err := folders.DeleteFolder(cmd.Args[0])
	if err == nil {
		fmt.Println("Folder removed.")
	}
	return err
}
