package repl

import (
	"fmt"
	"sort"
	"strings"

	"git.lyda.ie/pp/bulletin/dclish"
	"git.lyda.ie/pp/bulletin/pager"
)

var helpmap = map[string]string{
	"BULLETIN": `The BULLETIN utility permits a user to create a message for reading by
all users.  Users are notified upon logging in that new messages have
been added, and what the topic of the messages are.  Actual reading of
the messages is optional. (See the command SET READNEW for info on
automatic reading.)  Messages are automatically deleted when their
expiration date has passed.`,
	"FOLDERS": `All messages are divided into separate folders.  The default folder is
GENERAL.  New folders can be created by any user.  As an example, the
following creates a folder for GAMES related messages:

BULLETIN> CREATE GAMES
Enter a one line description of folder.
GAMES

To see the list of available folders, use DIRECTORY/FOLDERS.  To select
a specific folder, use the SELECT command.

If a user selects a folder and enters the SET READNEW command, that
user will be alerted of topics of new messages at login time, and will
then be given the option of reading them.  Similar to READNEW is SHOWNEW,
which displays the topics but doesn't prompt to read them.  Even less is
SET BRIEF, which will cause only a one line output indicating that there
are new messages in the folder.  There also is the SET NOTIFY option,
which will cause a message to be broadcast to a user's terminal alerting
the user that a new message has been added.  Any of these options can be
the default for the folder by using the /DEFAULT switch on the command.

A folder can be restricted to only certain users, if desired.  This is
done by specifying CREATE/PRIVATE.  Afterwards, access to the folder is
controlled by the creator by the SET [NO]ACCESS command.  If /SEMIPRIVATE
rather than /PRIVATE is specified, all users can read the messages in the
folder, but only those give access can add messages.

A folder can be converted into a remote folder using CREATE/NODE or SET
NODE.  A remote folder is one which points to a folder on a remote DECNET
node.  Messages added to a remote node are actually stored on the folder
on the remote node.  The BULLCP process (created by BULLETIN/STARTUP)
must be running on the remote node for this option to be used.

A folder can be specified as a SYSTEM folder, i.e. one in which SYSTEM/
SHUTDOWN/BROADCAST messages can be added.  By default, the GENERAL folder
is a SYSTEM folder (and cannot be changed).  One use for this is to create
a remote SYSTEM folder which is shared by all nodes, so that the GENERAL
folder is used for messages pertaining only to the local host, while the
remote folder is used for messages pertaining to all nodes.  Another
use is to create a folder for posting SYSTEM messages only meant for a
certain UIC group.  This is done by creating a PRIVATE SYSTEM folder, and
giving access to that UIC group.  Only users in that UIC group will see
the messages in that folder when they log in.`,
	"CTRL-C": `Except for when BULLETIN is awaiting input from the terminal, a
CTRL-C will cause BULLETIN to abort the execution of any command.  If
BULLETIN is waiting for terminal input, a CTRL-C will cause BULLETIN
to return to the BULLETIN> prompt.  If for some reason the user wishes
to suspend BULLETIN, CTRL-Y will usually do so.  However, this is not
always true, as BULLETIN will ignore the CTRL-Y if it has a data file
opened at the time. (Otherwise it would be possible to put the files
in a state such that they would be inaccessible by other users.)
`,
	"KEYPAD": `             +--------+--------+--------+--------+
             | PF1    | PF2    | PF3    | PF4    |
             |   GOLD |   HELP | EXTRACT|SHOW KEY|
             |        |ST NOKEY|  FILE  |SH KY/PR|
             |--------|--------|--------|--------|
             | 7      | 8      | 9      | --     |
             |  ADD   | REPLY  |  MAIL  |READ/NEW|
             | ADD/EDI|RP/ED/EX|M/NOHEAD|SHOW NEW|
             |--------|--------|--------|--------|
             | 4      | 5      | 6      | ,      |
             | CURRENT| RESPOND|  LAST  | DIR/NEW|
             |CURR/EDI|RS/ED/EX|        |  INDEX |
             |--------|--------|--------|--------|
             | 1      | 2      | 3      |ENTER   |
             |  BACK  |  PRINT |   DIR  |        |
             |  NEXT  |P/NONOTI|DIR/FOLD|        |
             |--------+--------|--------| ENTER  |
             | 0               | .      | SELECT |
             | SHOW FOLDER/FULL| DELETE |        |
             |    SHOW FLAGS   | UNDELE |        |
             +-----------------+--------+--------+`,
}

var subhelpmap = map[string]map[string]string{}

func generateHelp(hmap map[string]string, cmds dclish.Commands) {
	for c := range cmds {
		buf := &strings.Builder{}
		fmt.Fprint(buf, cmds[c].Description)
		if len(cmds[c].Flags) > 0 {
			flgs := make([]string, len(cmds[c].Flags))
			i := 0
			for flg := range cmds[c].Flags {
				flgs[i] = flg
				i++
			}
			sort.Strings(flgs)
			for i := range flgs {
				if strings.HasPrefix(cmds[c].Flags[flgs[i]].Description, "/") {
					fmt.Fprintf(buf, "\n\n%s", cmds[c].Flags[flgs[i]].Description)
				} else {
					fmt.Fprintf(buf, "\n\n%s\n\n%s", flgs[i], cmds[c].Flags[flgs[i]].Description)
				}
			}
		}
		hmap[c] = buf.String()
		if cmds[c].Commands != nil {
			subhelpmap[c] = map[string]string{}
			generateHelp(subhelpmap[c], cmds[c].Commands)
		}
	}
}

func init() {
	// Add all command help.
	generateHelp(helpmap, commands)

	// Add a list of topics.
	topics := make([]string, len(helpmap))
	i := 0
	maxlen := 0
	for topic := range helpmap {
		maxlen = max(len(topic), maxlen)
		topics[i] = topic
		i++
	}
	maxlen = maxlen + 2
	sort.Strings(topics)

	buf := &strings.Builder{}
	linelen := 2
	fmt.Fprint(buf,
		"\n\nThe following commands and topics are available for more help\n\n  ")
	pretty := map[string]string{
		"FOLDERS": "Folders",
		"CTRL-C":  "Ctrl-C",
		"KEYPAD":  "Keypad",
	}
	for i := range topics {
		linelen += maxlen
		if linelen > 78 {
			fmt.Fprint(buf, "\n  ")
			linelen = maxlen + 2
		}
		topic := pretty[topics[i]]
		if topic == "" {
			topic = topics[i]
		}
		fmt.Fprintf(buf, "%-*s", maxlen, topic)
	}
	fmt.Fprint(buf, "\n")
	helpmap["HELP"] += buf.String()
}

func findHelp(hmap map[string]string, args []string, fullcmd string) string {
	wordup := strings.ToUpper(args[0])
	helptext, ok := hmap[wordup]
	if !ok {
		possibles := []string{}
		for word := range hmap {
			if strings.HasPrefix(word, wordup) {
				possibles = append(possibles, word)
			}
		}
		switch len(possibles) {
		case 0:
			return fmt.Sprintf("ERROR: Topic not found: '%s'.\n", fullcmd)
		case 1:
			if len(args) == 2 {
				return findHelp(subhelpmap[possibles[0]], args[1:], strings.Join(args, " "))
			}
			return hmap[possibles[0]]
		default:
			return fmt.Sprintf("ERROR: Ambiguous topic '%s' (matches %s)\n",
				args[0], strings.Join(possibles, ", "))
		}
	}
	if len(args) == 2 {
		return findHelp(subhelpmap[wordup], args[1:], strings.Join(args, " "))
	}
	return helptext
}

// ActionHelp handles the `HELP` command.  This provides help taxt
// for each command.
//
// This originally existed as the subroutine HELP in bulletin8.for.
func ActionHelp(cmd *dclish.Command) error {
	if len(cmd.Args) == 0 {
		fmt.Printf("%s\n", helpmap["HELP"])
		return nil
	}

	pager.Pager(findHelp(helpmap, cmd.Args, cmd.Args[0]))
	return nil
}
