package repl

import (
	"errors"
	"fmt"
	"strconv"
	"strings"
	"time"

	"git.lyda.ie/pp/bulletin/ask"
	"git.lyda.ie/pp/bulletin/dclish"
	"git.lyda.ie/pp/bulletin/editor"
	"git.lyda.ie/pp/bulletin/folders"
	"git.lyda.ie/pp/bulletin/pager"
	"git.lyda.ie/pp/bulletin/storage"
	"git.lyda.ie/pp/bulletin/this"
)

// ActionDirectory handles the `DIRECTORY` command.  This lists all the
// messages in the current folder.
//
// This originally existed as the subroutines DIRECTORY in bulletin0.for
// and DIRECTORY_FOLDERS in bulletin5.for.
func ActionDirectory(cmd *dclish.Command) error {
	showExpiration := false
	if cmd.Flags["/EXPIRATION"].Value == "true" {
		showExpiration = true
	}
	if len(cmd.Args) == 1 {
		folder, err := folders.ValidFolder(cmd.Args[0])
		if err != nil {
			fmt.Println("Folder does not exist.")
			return nil
		}
		if !folders.IsFolderReadable(folder.Name, this.User.Login) {
			fmt.Println("No permission to access folder.")
			return nil
		}
		this.Folder = folder
		this.ReadFirstCall = true
	}
	msgs, err := folders.ListMessages(this.Folder.Name)
	if err != nil {
		return err
	}
	if len(msgs) == 0 {
		fmt.Println("There are no messages present.")
		return nil
	}
	buf := strings.Builder{}
	buf.WriteString(fmt.Sprintf("%4s %-43s %-12s %-10s\n", "#", "Subject", "From", "Date"))
	for _, msg := range msgs {
		buf.WriteString(fmt.Sprint(msg.OneLine(showExpiration)))
	}
	pager.Pager(buf.String())

	return nil
}

// ActionAdd handles the `ADD` command.  This adds a message to a folder.
//
// This originally existed as the subroutine ADD in bulletin.for.
func ActionAdd(cmd *dclish.Command) error {
	ctx := storage.Context()

	optAll := 0
	optBell := 0
	optBroadcast := 0
	optExpiration := &time.Time{}
	optExtract := 0
	optFolder := []string{}
	optIndent := 0
	optPermanent := 0
	optShutdown := 0
	optSignature := 0
	optSubject := ""
	optSystem := 0

	if cmd.Flags["/ALL"].Value == "true" {
		optAll = 1
	}
	if cmd.Flags["/ALL"].Set {
		fmt.Printf("/ALL is not yet implemented - you set it to %d\n", optAll)
	}
	if cmd.Flags["/BELL"].Value == "true" {
		optBell = 1
	}
	if cmd.Flags["/BELL"].Set {
		fmt.Printf("/BELL is not yet implemented - you set it to %d\n", optBell)
	}
	if cmd.Flags["/BROADCAST"].Value == "true" {
		optBroadcast = 1
	}
	if cmd.Flags["/BROADCAST"].Set {
		fmt.Printf("/BROADCAST is not yet implemented - you set it to %d\n", optBroadcast)
	}
	if cmd.Flags["/EXPIRATION"].Value != "" {
		// dd-mmm-yyyy, or delta time: dddd
		exp, err := time.Parse("2006-01-02", cmd.Flags["/EXPIRATION"].Value)
		if err != nil {
			days, err := strconv.Atoi(cmd.Flags["/EXPIRATION"].Value)
			if err != nil {
				optExpiration = nil
			}
			exp := time.Now().AddDate(0, 0, days).UTC()
			optExpiration = &exp
		} else {
			optExpiration = &exp
		}
	}
	if cmd.Flags["/EXTRACT"].Value == "true" {
		optExtract = 1
	}
	if cmd.Flags["/EXTRACT"].Set {
		fmt.Printf("/EXTRACT is not yet implemented - you set it to %d\n", optExtract)
	}
	if cmd.Flags["/FOLDER"].Value != "" {
		optFolder = strings.Split(cmd.Flags["/FOLDER"].Value, ",")
	}
	if cmd.Flags["/INDENT"].Value == "true" {
		optIndent = 1
	}
	if cmd.Flags["/INDENT"].Set {
		fmt.Printf("/INDENT is not yet implemented - you set it to %d\n", optIndent)
	}
	if cmd.Flags["/PERMANENT"].Value == "true" {
		optPermanent = 1
	}
	if cmd.Flags["/SHUTDOWN"].Value == "true" {
		optShutdown = 1
	}
	if cmd.Flags["/SIGNATURE"].Value == "true" {
		optSignature = 1
	}
	if cmd.Flags["/SIGNATURE"].Set {
		fmt.Printf("/SIGNATURE is not yet implemented - you set it to %d\n", optSignature)
	}
	if cmd.Flags["/SUBJECT"].Value != "" {
		optSubject = cmd.Flags["/SUBJECT"].Value
	}
	if cmd.Flags["/SYSTEM"].Value == "true" {
		optSystem = 1
	}
	if cmd.Flags["/SYSTEM"].Set {
		fmt.Printf("/SYSTEM is not yet implemented - you set it to %d\n", optSystem)
	}

	if len(optFolder) == 0 {
		optFolder = []string{this.Folder.Name}
	}
	for i := range optFolder {
		f, _ := this.Q.FindFolderExact(ctx, optFolder[i])
		if f.Name == "" {
			return fmt.Errorf("Folder '%s' does not exist", optFolder[i])
		}
		if f.Visibility != 0 && this.User.Admin == 0 && this.User.Login != f.Owner {
			return fmt.Errorf("Folder '%s' is not accessible", optFolder[i])
		}
	}

	if optSubject == "" {
		optSubject, _ = ask.GetLine("Enter subject of message: ")
		if optSubject == "" {
			return errors.New("Must enter a subject")
		}
	}
	if optShutdown == 1 && this.User.Admin == 0 {
		return errors.New("Must be an admin to set shutdown")
	}

	message, err := editor.Editor(
		fmt.Sprintf("Enter message for '%s'...", optSubject),
		"Edit message",
		"")
	if err != nil {
		return err
	}
	for i := range optFolder {
		err = folders.CreateMessage(this.User.Login, optSubject, message,
			optFolder[i], optPermanent, optShutdown, optExpiration)
	}
	return nil
}

// ActionCurrent handles the `CURRENT` command.  This shows the head of
// the current message.
//
// This originally existed as... not sure.
func ActionCurrent(_ *dclish.Command) error {
	msg, err := folders.GetMessage(this.User.Login, this.Folder.Name, this.MsgID)
	if err != nil {
		return err
	}
	lines := strings.Split(msg.String(), "\n")
	if len(lines) > 10 {
		lines = lines[:10]
	}
	fmt.Printf("%s\n", strings.Join(lines, "\n"))
	return nil
}

// ActionBack handles the `BACK` command.  Goes back and shows the previous
// message.
//
// This originally existed as... not sure.
func ActionBack(_ *dclish.Command) error {
	msgid := folders.PrevMsgid(this.User.Login, this.Folder.Name, this.MsgID)
	if msgid == 0 {
		fmt.Println("No previous messages")
		return nil
	}
	msg, err := folders.GetMessage(this.User.Login, this.Folder.Name, msgid)
	if err != nil {
		return err
	}
	if pager.Pager(msg.String()) {
		this.MsgID = msgid
		folders.MarkSeen([]int64{msgid})
	}
	return nil
}

// ActionChange handles the `CHANGE` command.  This replaces or modifies
// an existing message.
//
// This originally existed as... not sure.
func ActionChange(cmd *dclish.Command) error {
	var err error
	isAdmin := this.User.Admin == 1

	optAll := false
	if cmd.Flags["/ALL"].Value == "true" {
		if !isAdmin {
			return errors.New("Must be admin")
		}
		optAll = true
	}

	optExpiration := &time.Time{}
	if cmd.Flags["/EXPIRATION"].Value != "" {
		// dd-mmm-yyyy, or delta time: dddd
		exp, err := time.Parse("2006-01-02", cmd.Flags["/EXPIRATION"].Value)
		if err != nil {
			days, err := strconv.Atoi(cmd.Flags["/EXPIRATION"].Value)
			if err != nil {
				optExpiration = nil
			}
			exp := time.Now().AddDate(0, 0, days)
			optExpiration = &exp
		} else {
			optExpiration = &exp
		}
	}

	optGeneral := false
	if cmd.Flags["/GENERAL"].Set && this.Folder.Name != "GENERAL" {
		return errors.New("Can only be used in the GENERAL folder")
	}
	if cmd.Flags["/GENERAL"].Value == "true" {
		if !isAdmin {
			return errors.New("Must be admin")
		}
		optGeneral = true
	}
	if cmd.Flags["/GENERAL"].Set && !optGeneral {
		return errors.New("Can't specify /NOGENERAL - see /SYSTEM")
	}

	optNew := false
	if cmd.Flags["/NEW"].Value == "true" {
		optNew = true
	}

	msgids := []int64{this.MsgID}
	if cmd.Flags["/NUMBER"].Set && optAll {
		return errors.New("Can't set /ALL and /NUMBER")
	}
	if cmd.Flags["/NUMBER"].Set && cmd.Flags["/NUMBER"].Value == "" {
		return errors.New("Must supply message number(s) if /NUMBER is set")
	}
	if cmd.Flags["/NUMBER"].Value != "" {
		msgids, err = ParseNumberList(cmd.Flags["/NUMBER"].Value)
		if err != nil {
			return err
		}
	}

	optPermanent := false
	if cmd.Flags["/PERMANENT"].Value == "true" {
		optPermanent = true
	}

	optShutdown := false
	if cmd.Flags["/SHUTDOWN"].Value == "true" {
		if !isAdmin {
			return errors.New("Must be admin")
		}
		optShutdown = true
	}

	optSubject := ""
	if cmd.Flags["/SUBJECT"].Set && cmd.Flags["/SUBJECT"].Value == "" {
		return errors.New("Must supply subject text if /SUBJECT is set")
	}
	if cmd.Flags["/SUBJECT"].Value != "" {
		optSubject = cmd.Flags["/SUBJECT"].Value
	}

	optSystem := false
	if cmd.Flags["/SYSTEM"].Set {
		if !isAdmin {
			return errors.New("Must be admin")
		}
		if this.Folder.Name != "GENERAL" {
			return errors.New("Can only be used in the GENERAL folder")
		}
	}
	if cmd.Flags["/SYSTEM"].Value == "true" {
		optSystem = true
	}
	if cmd.Flags["/SYSTEM"].Set && !optSystem {
		return errors.New("Can't specify /NOSYSTEM - see /GENERAL")
	}

	optText := false
	if cmd.Flags["/TEXT"].Value == "true" {
		optText = true
	}

	// Sanity checking.
	if optSystem && optGeneral {
		return errors.New("Can't specify /SYSTEM and /GENERAL")
	}
	if optNew && !optText {
		return errors.New("Must specify /TEXT if you specify /NEW")
	}
	ctx := storage.Context()
	if optAll {
		if optText {
			return errors.New("Can't specify /TEXT with /ALL")
		}
		msgids, err = this.Q.ListMessageIDs(ctx, this.Folder.Name)
		if err != nil {
			return err
		}
	}

	isFolderOwner := folders.IsFolderOwner(this.Folder.Name, this.User.Login)
	for i := range msgids {
		msg, err := folders.GetMessage(this.User.Login, this.Folder.Name, msgids[i])
		if err != nil {
			return err
		}
		if msg.ID == 0 {
			return fmt.Errorf("Failed to retrieve message number %d", msgids[i])
		}
		if !(isAdmin || isFolderOwner || msg.Author == this.User.Login) {
			return errors.New("Only admin or folder owner or author can change message")
		}
		if optSystem {
			msg.System = 1
		} else if optGeneral {
			msg.System = 0
		}
		if cmd.Flags["/EXPIRATION"].Set {
			msg.Expiration = *optExpiration
		}
		if optPermanent {
			msg.Permanent = 1
		}
		if optShutdown {
			msg.Shutdown = 1
		}
		if cmd.Flags["/SUBJECT"].Set {
			msg.Subject = optSubject
		}
		if optText {
			if optNew {
				msg.Message = ""
			}
			msg.Message, err = editor.Editor(
				fmt.Sprintf("Enter message for '%s'...", msg.Subject),
				"Edit message",
				msg.Message)
		}
		err = this.Q.UpdateMessage(ctx, storage.UpdateMessageParams{
			System:     msg.System,
			Expiration: msg.Expiration,
			Permanent:  msg.Permanent,
			Shutdown:   msg.Shutdown,
			Subject:    msg.Subject,
			Message:    msg.Message,
			ID:         msg.ID,
		})
		if err != nil {
			return err
		}
	}

	return nil
}

// ActionFirst handles the `FIRST` command.  Prints the first message in
// a folder.
//
// This originally existed as... not sure.
func ActionFirst(_ *dclish.Command) error {
	msgid := folders.FirstMessage(this.Folder.Name)
	if msgid == 0 {
		fmt.Println("No messages in folder")
		return nil
	}
	this.MsgID = msgid
	msg, err := folders.GetMessage(this.User.Login, this.Folder.Name, msgid)
	if err != nil {
		return err
	}
	if pager.Pager(msg.String()) {
		folders.MarkSeen([]int64{msgid})
	}
	return nil
}

// ActionLast handles the `LAST` command. Prints the last message.
//
// This originally existed as... not sure.
func ActionLast(_ *dclish.Command) error {
	msgid := folders.LastMessage(this.Folder.Name)
	if msgid == 0 {
		fmt.Println("No messages in folder")
		return nil
	}
	this.MsgID = msgid
	msg, err := folders.GetMessage(this.User.Login, this.Folder.Name, msgid)
	if err != nil {
		return err
	}
	if pager.Pager(msg.String()) {
		folders.MarkSeen([]int64{msgid})
	}
	return nil
}

// ActionNext handles the `NEXT` command.  Shows the next message.
//
// This originally existed as... not sure.
func ActionNext(_ *dclish.Command) error {
	msgid := folders.NextMsgid(this.User.Login, this.Folder.Name, this.MsgID)
	if msgid == 0 {
		fmt.Println("No next messages")
		return nil
	}
	msg, err := folders.GetMessage(this.User.Login, this.Folder.Name, msgid)
	if err != nil {
		return err
	}
	pager.Pager(msg.String())
	if pager.Pager(msg.String()) {
		folders.MarkSeen([]int64{msgid})
		this.MsgID = msgid
	}
	return nil
}

// ActionPrint handles the `PRINT` command.  This sends the message in
// a way that the terminal will redirect it to an attached printer.  Not
// many (any?) terminals support it now.
//
// This originally existed as the subroutine PRINT in bulletin1.for.
func ActionPrint(cmd *dclish.Command) error {
	all := false
	if cmd.Flags["/ALL"].Value == "true" {
		all = true
	}

	ctx := storage.Context()
	msgids := []int64{this.MsgID}
	var err error
	if len(cmd.Args) == 1 {
		if all {
			return errors.New("Can't provide a message list and /ALL")
		}
		msgids, err = ParseNumberList(cmd.Args[0])
		if err != nil {
			return err
		}
	} else if all {
		msgids, err = this.Q.ListMessageIDs(ctx, this.Folder.Name)
	}
	print("\033[5i")
	for _, msgid := range msgids {
		msg, err := folders.GetMessage(this.User.Login, this.Folder.Name, msgid)
		if err != nil {
			fmt.Printf("Message %d not found.\n", msgid)
		} else {
			fmt.Print(msg.String())
		}
		print("\n\v")
	}
	print("\033[4i")
	return nil
}

// ActionRead handles the `READ` command.  Increment (except the first
// time it's a called in a folder) and reads the message.
//
// This originally existed as the subroutine READ_MSG in bulletin1.for.
func ActionRead(cmd *dclish.Command) error {
	defer func() { this.ReadFirstCall = false }()
	msgid := this.MsgID
	if !this.ReadFirstCall && len(cmd.Args) == 0 {
		msgid = folders.NextMsgid(this.User.Login, this.Folder.Name, msgid)
		if msgid < this.MsgID {
			fmt.Println("No more unread messages.")
			return nil
		}
	} else if len(cmd.Args) == 1 {
		var err error
		msgid, err = strconv.ParseInt(cmd.Args[0], 10, 64)
		if err != nil {
			return err
		}
	}
	this.MsgID = msgid
	msg, err := folders.GetMessage(this.User.Login, this.Folder.Name, msgid)
	if err != nil {
		return err
	}
	if pager.Pager(msg.String()) {
		folders.MarkSeen([]int64{msgid})
	}
	return nil
}

// ActionReply handles the `REPLY` command.  Used to create a reply to
// a message.
//
// This originally existed as the subroutine REPLY in bulletin.for.
func ActionReply(cmd *dclish.Command) error {
	extract := true
	if cmd.Flags["/EXTRACT"].Value == "false" {
		extract = false
	}
	indent := true
	if cmd.Flags["/INDENT"].Value == "false" {
		indent = false
	}
	original, err := folders.GetMessage(this.User.Login, this.Folder.Name, this.MsgID)
	origMsg := ""
	if extract {
		if indent {
			origMsg = "> " + strings.Join(strings.Split(original.Message, "\n"), "\n> ")
		} else {
			origMsg = original.Message
		}
	}

	subject := "Re: " + original.Subject
	message, err := editor.Editor(
		fmt.Sprintf("Enter message for '%s'...", subject),
		"Edit message",
		origMsg)
	if err != nil {
		fmt.Printf("ERROR: Editor failure (%s).\n", err)
		return nil
	}
	err = folders.CreateMessage(this.User.Login, subject,
		message, this.Folder.Name, 0, 0, nil)
	if err != nil {
		fmt.Printf("ERROR: CreateMessage failure (%s).\n", err)
		return nil
	}
	return nil
}

// ActionSeen handles the `SEEN` command.  Marks messages as seen.
//
// This originally existed as... not sure.
func ActionSeen(cmd *dclish.Command) error {
	var err error
	msgids := []int64{this.MsgID}
	if len(cmd.Args) == 1 {
		msgids, err = ParseNumberList(cmd.Args[0])
		if err != nil {
			return err
		}
	}
	err = folders.MarkSeen(msgids)
	if err != nil {
		fmt.Printf("ERROR: %s.\n", err)
	}
	return nil
}

// ActionUnseen handles the `UNSEEN` command. Marks messages as unseen.
//
// This originally existed as... not sure.
func ActionUnseen(cmd *dclish.Command) error {
	var err error
	msgids := []int64{this.MsgID}
	if len(cmd.Args) == 1 {
		msgids, err = ParseNumberList(cmd.Args[0])
		if err != nil {
			return err
		}
	}
	err = folders.MarkUnseen(msgids)
	if err != nil {
		return err
	}
	return nil
}

// ActionDelete handles the `DELETE` command.  This deletes a message.
// As of now this is permanent.  It would need to be changed to support
// UNDELETE.
//
// This originally existed as the subroutine DELETE_MSG in bulletin0.for.
func ActionDelete(cmd *dclish.Command) error {
	var err error

	all := false
	if cmd.Flags["/ALL"].Value == "true" {
		all = true
	}
	if all {
		if len(cmd.Args) == 1 {
			fmt.Println("ERROR: Can't specify both message numbers and /ALL flag.")
			return nil
		}
		if this.User.Admin == 0 {
			return errors.New("Must be admin to use /ALL")
		}
		err := folders.DeleteAllMessages()
		if err != nil {
			return err
		}
		return nil
	}

	msgids := []int64{this.MsgID}
	if len(cmd.Args) == 1 {
		msgids, err = ParseNumberList(cmd.Args[0])
		if err != nil {
			return err
		}
	}
	if this.User.Admin == 0 && !folders.WroteAllMessages(this.User.Login, msgids) {
		return errors.New("Can't delete messages you haven't written")
	}
	err = folders.DeleteMessages(msgids)
	if err != nil {
		return err
	}
	return nil
}

// ActionMark handles the `MARK` command.  This sets a MARK on messages.
//
// This originally existed as... not sure.
func ActionMark(cmd *dclish.Command) error {
	var err error
	msgids := []int64{this.MsgID}
	if len(cmd.Args) == 1 {
		msgids, err = ParseNumberList(cmd.Args[0])
		if err != nil {
			return err
		}
	}
	err = folders.SetMark(msgids)
	if err != nil {
		fmt.Printf("ERROR: %s.\n", err)
	}
	return nil
}

// ActionUnmark handles the `UNMARK` command.  This removes a MARK on
// messages.
//
// This originally existed as... not sure.
func ActionUnmark(cmd *dclish.Command) error {
	var err error
	msgids := []int64{this.MsgID}
	if len(cmd.Args) == 1 {
		msgids, err = ParseNumberList(cmd.Args[0])
		if err != nil {
			return err
		}
	}
	err = folders.UnsetMark(msgids)
	if err != nil {
		fmt.Printf("ERROR: %s.\n", err)
	}
	return nil
}

// ActionSearch handles the `SEARCH` command.  This will show all messages
// matching a search term.
//
// See subtoutines SEARCH and GET_SEARCH in bulletin2.for for the
// original implementation.
func ActionSearch(cmd *dclish.Command) error {
	ctx := storage.Context()
	var err error
	optFolders := []string{this.Folder.Name}
	if cmd.Flags["/FOLDER"].Value != "" {
		optFolders = strings.Split(strings.ToUpper(cmd.Flags["/FOLDER"].Value), ",")
		for i := range optFolders {
			folder, _ := this.Q.FindFolderExact(ctx, optFolders[i])
			if folder.Name != "" {
				return fmt.Errorf("Folder '%s' not found", optFolders[i])
			}
			if !folders.IsFolderReadable(optFolders[i], this.User.Login) {
				return fmt.Errorf("Folder '%s' is not accessible", optFolders[i])
			}
		}
	}

	optReply := false
	if cmd.Flags["/REPLY"].Value == "true" {
		optReply = true
	}

	optReverse := false
	if cmd.Flags["/REVERSE"].Value == "true" {
		optReverse = true
	}

	optStart := int64(-1) // -1 means first message.
	if optReverse {
		optStart = 0 // 0 means last message.
	}
	if cmd.Flags["/START"].Set {
		optStart, err = strconv.ParseInt(cmd.Flags["/START"].Value, 10, 64)
		if err != nil {
			return err
		}
		if optStart < 1 {
			return errors.New("/START must be 1 or larger")
		}
	}

	optSubject := false
	if cmd.Flags["/SUBJECT"].Value == "true" {
		optSubject = true
	}

	subject := ""
	if optReply {
		if optSubject {
			return errors.New("Can't specify /REPLY and /SUBJECT")
		}
		msg, err := this.Q.ReadMessage(ctx, this.Folder.Name, this.MsgID)
		if err != nil {
			return err
		}
		subject = "Re: " + msg.Subject
	}

	allMsgs := []storage.Message{}
	var msgs []storage.Message
	var start int64
	for i := range optFolders {
		switch optStart {
		case -1:
			start = 1
		case 0:
			start, err := this.Q.LastMsgidIgnoringSeen(ctx, optFolders[i])
			if err != nil || start == 0 {
				continue
			}
		default:
			start = optStart
		}
		if optReply {
			if optReverse {
				msgs, err = this.Q.SearchReplyReverse(ctx,
					storage.SearchReplyReverseParams{
						Column1: nullStr(cmd.Args[0]),
						Subject: subject,
						ID:      start,
						Folder:  optFolders[i],
					})
			} else {
				msgs, err = this.Q.SearchReply(ctx,
					storage.SearchReplyParams{
						Column1: nullStr(cmd.Args[0]),
						Subject: subject,
						ID:      start,
						Folder:  optFolders[i],
					})
			}
		} else if optSubject {
			if optReverse {
				msgs, err = this.Q.SearchSubjectReverse(ctx, nullStr(cmd.Args[0]), start, optFolders[i])
			} else {
				msgs, err = this.Q.SearchSubject(ctx, nullStr(cmd.Args[0]), start, optFolders[i])
			}
		} else {
			if optReverse {
				msgs, err = this.Q.SearchReverse(ctx, nullStr(cmd.Args[0]), start, optFolders[i])
			} else {
				msgs, err = this.Q.Search(ctx, nullStr(cmd.Args[0]), start, optFolders[i])
			}
		}
		if err != nil {
			continue
		}
		if len(allMsgs)+len(msgs) > 100 {
			fmt.Println("Too many messages match; narrow search term.")
			return nil
		}
		allMsgs = append(allMsgs, msgs...)
	}

	if len(allMsgs) == 0 {
		fmt.Println("No messages found.")
		return nil
	}
	buf := strings.Builder{}
	for _, msg := range allMsgs {
		buf.WriteString(msg.String())
		buf.WriteString("\n\n")
	}
	pager.Pager(buf.String())

	return nil
}
