// Package repl defines the [dclish] commands and implements the actions
// and defines the REPL to get the commands.
package repl

import (
	"fmt"
	"os"
	"path"
	"strings"
	"unicode"

	"git.lyda.ie/pp/bulletin/dclish"
	"git.lyda.ie/pp/bulletin/this"
	"github.com/adrg/xdg"
	"github.com/chzyer/readline"
)

// Loop is the main event loop.
func Loop() error {
	completer := dclish.NewCompleter(commands)
	histdir := path.Join(xdg.ConfigHome, "BULLETIN")
	os.MkdirAll(histdir, 0700)
	histfile := path.Join(histdir, fmt.Sprintf("%s.history", this.User.Login))
	rl, err := readline.NewEx(
		&readline.Config{
			Prompt:            "BULLETIN> ",
			HistoryFile:       histfile,
			AutoComplete:      completer,
			InterruptPrompt:   "^C",
			EOFPrompt:         "EXIT",
			HistorySearchFold: true,
		})
	if err != nil {
		return err
	}
	defer rl.Close()

	missing := []string{}
	for c := range commands {
		if commands[c].Action == nil {
			missing = append(missing, c)
		}
		if len(commands[c].Commands) > 0 {
			for subc := range commands[c].Commands {
				if commands[c].Commands[subc].Action == nil {
					missing = append(missing, c+" "+subc)
				}
			}
		}
	}
	if len(missing) != 0 {
		fmt.Printf("ERROR: some commands lack actions: %s.\n",
			strings.Join(missing, ", "))
	}

	for {
		this.ShowBroadcast()
		line, err := rl.Readline()
		if err != nil {
			if err.Error() == "Interrupt" {
				commands.ParseAndRun("QUIT")
			} else if err.Error() == "EOF" {
				commands.ParseAndRun("EXIT")
			}
			return err
		}
		if len(line) == 0 {
			continue
		}
		prependRead := false
		for _, r := range line {
			if unicode.IsDigit(r) {
				prependRead = true
				break
			}
			if !unicode.IsSpace(r) {
				break
			}
		}
		if prependRead {
			line = "READ " + line
		}
		err = commands.ParseAndRun(line)
		if err != nil {
			fmt.Printf("ERROR: %s.\n", err)
		}
	}
}
