package repl

import (
	"errors"
	"fmt"
	"strconv"

	"git.lyda.ie/pp/bulletin/dclish"
	"git.lyda.ie/pp/bulletin/folders"
	"git.lyda.ie/pp/bulletin/storage"
	"git.lyda.ie/pp/bulletin/this"
)

func setAlert(cmd *dclish.Command, alert int64) error {
	optAll := cmd.Flags["/ALL"].Value == "true"

	optPerm := false
	if f, ok := cmd.Flags["/PERMANENT"]; ok {
		optPerm = f.Value == "true"
	}

	folder := this.Folder
	if cmd.Flags["/FOLDER"].Value != "" {
		folder = folders.FindFolder(cmd.Flags["/FOLDER"].Value)
	}
	if folder.Name == "" {
		return errors.New("Folder does not exist")
	}

	if optAll || optPerm {
		if !folders.IsFolderOwner(folder.Name, this.User.Login) {
			return errors.New("Not an admin or folder owner")
		}
	}

	ctx := storage.Context()
	if optAll && optPerm {
		return this.Q.UpdateFolderAlert(ctx, alert+3, folder.Name)
	}
	if optAll {
		return this.Q.UpdateFolderAlert(ctx, alert, folder.Name)
	}
	return this.Q.UpdateUserAlert(ctx, this.User.Login, folder.Name, alert)
}

// ActionSet handles the `SET` command.  This command has multiple
// subcommands.  This just prints the help you'd see with "HELP SET".
func ActionSet(cmd *dclish.Command) error {
	fmt.Println(cmd.Description)
	return nil
}

// ActionSetNoaccess handles the `SET ACCESS` command.
//
// This originally existed as the subroutine SET_ACCESS in bulletin5.for.
func ActionSetNoaccess(cmd *dclish.Command) error {
	ctx := storage.Context()
	login := cmd.Args[0]
	folder := this.Folder
	if len(cmd.Args) == 2 {
		folder = folders.FindFolder(cmd.Args[1])
		if folder.Name == "" {
			return errors.New("Folder not found")
		}
	}
	if this.User.Admin == 0 || folder.Owner != this.User.Login {
		return errors.New("Must be an admin or folder owner")
	}
	this.Q.DeleteFolderAccess(ctx, login, folder.Name)
	return nil
}

// ActionSetAccess handles the `SET ACCESS` command.
//
// This originally existed as the subroutine SET_ACCESS in bulletin5.for.
func ActionSetAccess(cmd *dclish.Command) error {
	ctx := storage.Context()
	optAll := cmd.Flags["/ALL"].Set
	if optAll {
		if cmd.Flags["/ALL"].Value != "true" {
			return errors.New("Flag '/NOALL' not recognised")
		}
	}
	optRead := cmd.Flags["/READ"].Set
	if optRead {
		if cmd.Flags["/READ"].Value != "true" {
			return errors.New("Flag '/READ' not recognised")
		}
	}
	if optAll {
		if len(cmd.Args) > 1 {
			return errors.New("Too many arguments for /ALL")
		}
		folder := this.Folder
		if len(cmd.Args) == 1 {
			folder = folders.FindFolder(cmd.Args[0])
			if folder.Name == "" {
				return errors.New("Folder not found")
			}
		}
		if this.User.Admin == 0 || folder.Owner != this.User.Login {
			return errors.New("Must be an admin or folder owner")
		}
		visibility := int64(2)
		if optRead {
			visibility = 1
		}
		this.Q.UpdateFolderVisibility(ctx, visibility, folder.Name)
		return nil
	}
	if len(cmd.Args) == 0 {
		return errors.New("Must supply a user login to set access")
	}
	login := cmd.Args[0]
	folder := this.Folder
	if len(cmd.Args) == 2 {
		folder = folders.FindFolder(cmd.Args[1])
		if folder.Name == "" {
			return errors.New("Folder not found")
		}
	}
	if this.User.Admin == 0 || folder.Owner != this.User.Login {
		return errors.New("Must be an admin or folder owner")
	}
	visibility := int64(2)
	if optRead {
		visibility = 1
	}
	this.Q.UpdateFolderAccess(ctx, login, folder.Name, visibility)
	return nil
}

// ActionSetAlways handles the `SET ALWAYS` command.
func ActionSetAlways(_ *dclish.Command) error {
	ctx := storage.Context()
	this.Q.UpdateFolderAlways(ctx, 1, this.Folder.Name)
	return nil
}

// ActionSetNoalways handles the `SET NOALWAYS` command.
func ActionSetNoalways(_ *dclish.Command) error {
	ctx := storage.Context()
	this.Q.UpdateFolderAlways(ctx, 0, this.Folder.Name)
	return nil
}

// ActionSetBrief handles the `SET BRIEF` command.
func ActionSetBrief(cmd *dclish.Command) error {
	return setAlert(cmd, storage.AlertBrief)
}

// ActionSetNobrief handles the `SET NOBRIEF` command.
func ActionSetNobrief(cmd *dclish.Command) error {
	return setAlert(cmd, storage.AlertNone)
}

// ActionSetDefaultExpire handles the `SET DEFAULT_EXPIRE` command.
func ActionSetDefaultExpire(cmd *dclish.Command) error {
	if this.User.Admin == 0 {
		return errors.New("You are not an admin")
	}
	value, err := strconv.ParseInt(cmd.Args[0], 10, 64)
	if err != nil {
		return err
	}
	ctx := storage.Context()
	return this.Q.UpdateDefaultExpire(ctx, value)
}

// ActionSetExpireLimit handles the `SET EXPIRE_LIMIT` command.
func ActionSetExpireLimit(cmd *dclish.Command) error {
	if this.User.Admin == 0 {
		return errors.New("You are not an admin")
	}
	value, err := strconv.ParseInt(cmd.Args[0], 10, 64)
	if err != nil {
		return err
	}
	ctx := storage.Context()
	return this.Q.UpdateExpireLimit(ctx, value)
}

// ActionSetPromptExpire handles the `SET PROMPT_EXPIRE` command.
func ActionSetPromptExpire(_ *dclish.Command) error {
	ctx := storage.Context()
	this.Q.UpdateUserPrompt(ctx, 1, this.User.Login)
	return nil
}

// ActionSetNoPromptExpire handles the `SET NOPROMPT_EXPIRE` command.
func ActionSetNoPromptExpire(_ *dclish.Command) error {
	ctx := storage.Context()
	this.Q.UpdateUserPrompt(ctx, 0, this.User.Login)
	return nil
}

// ActionSetReadNew handles the `SET READNEW` command.
func ActionSetReadNew(cmd *dclish.Command) error {
	return setAlert(cmd, storage.AlertReadNew)
}

// ActionSetNoReadNew handles the `SET READNEW` command.
func ActionSetNoReadNew(cmd *dclish.Command) error {
	return setAlert(cmd, storage.AlertNone)
}

// ActionSetShowNew handles the `SET SHOWNEW` command.
func ActionSetShowNew(cmd *dclish.Command) error {
	return setAlert(cmd, storage.AlertShowNew)
}

// ActionSetNoShowNew handles the `SET SHOWNEW` command.
func ActionSetNoShowNew(cmd *dclish.Command) error {
	return setAlert(cmd, storage.AlertNone)
}

// ActionSetSystem handles the `SET SYSTEM` command.
func ActionSetSystem(_ *dclish.Command) error {
	if this.User.Admin == 0 {
		return errors.New("You are not an admin")
	}
	ctx := storage.Context()
	this.Q.UpdateFolderSystem(ctx, 1, this.Folder.Name)
	return nil
}

// ActionSetNosystem handles the `SET SYSTEM` command.
func ActionSetNosystem(_ *dclish.Command) error {
	if this.User.Admin == 0 {
		return errors.New("You are not an admin")
	}
	if this.Folder.Name == "GENERAL" {
		fmt.Println("Can't remove SYSTEM from the GENERAL folder.")
		return nil
	}
	ctx := storage.Context()
	this.Q.UpdateFolderSystem(ctx, 1, this.Folder.Name)
	return nil
}
