package repl

import (
	"fmt"
	"strings"

	"github.com/carlmjohnson/versioninfo"

	"git.lyda.ie/pp/bulletin/ask"
	"git.lyda.ie/pp/bulletin/dclish"
	"git.lyda.ie/pp/bulletin/folders"
	"git.lyda.ie/pp/bulletin/storage"
	"git.lyda.ie/pp/bulletin/this"
)

// ActionShow handles the `SHOW` command.
func ActionShow(cmd *dclish.Command) error {
	fmt.Println(cmd.Description)
	return nil
}

// ActionShowFlags handles the `SHOW FLAGS` command.
func ActionShowFlags(_ *dclish.Command) error {
	flagset := false
	fmt.Printf("For the selected folder %s:\n", this.Folder.Name)
	if this.Folder.Alert != 0 {
		fmt.Printf("  %s is set.\n",
			strings.ToUpper(storage.AlertString(this.Folder.Alert)))
		flagset = true
	}
	if !flagset {
		fmt.Println("  No flags are set.")
	}
	return nil
}

// ActionShowFolder handles the `SHOW FOLDER` command.  This is based on
// `SHOW_FOLDER` in bulletin5.for.
func ActionShowFolder(cmd *dclish.Command) error {
	folder := this.Folder
	if len(cmd.Args) == 1 {
		folder = folders.FindFolder(cmd.Args[0])
	}
	if folder.Name == "" {
		fmt.Println("ERROR: Specified folder was not found.")
	}

	full := false
	if cmd.Flags["/FULL"].Value == "true" {
		if this.User.Admin == 0 {
			fmt.Println("ERROR: You are not an admin.")
			return nil
		}
		full = true
	}
	fmt.Printf("Settings for %s (%s).\n", folder.Name, folder.Owner)
	if full {
		switch folder.Visibility {
		case folders.FolderPublic:
			fmt.Println("  Folder is a public folder.")
		case folders.FolderSemiPrivate:
			fmt.Println("  Folder is a semi-private folder.")
		case folders.FolderPrivate:
			fmt.Println("  Folder is a private folder.")
		}
	}
	switch folder.Expire {
	case -1:
		fmt.Println("  Default expiration is permanent.")
	case 0:
		fmt.Println("  No default expiration set.")
	default:
		fmt.Printf("  Default expiration is %d days.\n", folder.Expire)
	}
	if folder.System != 0 {
		fmt.Println("  SYSTEM has been set.")
	}
	if folder.Always != 0 {
		fmt.Println("  ALWAYS has been set.")
	}
	if this.Folder.Alert != 0 {
		fmt.Printf("  %s is set.\n",
			strings.ToUpper(storage.AlertString(this.Folder.Alert)))
	}

	return nil
}

// ActionShowNew handles the `SHOW NEW` command.
func ActionShowNew(_ *dclish.Command) error {
	this.ShowAlerts(false)
	return nil
}

// ActionShowPrivileges handles the `SHOW PRIVILEGES` command.
func ActionShowPrivileges(_ *dclish.Command) error {
	fmt.Printf("Account privileges for %s:\n", this.User.Login)
	fmt.Printf("  Admin     : %s\n", ask.YesNo(this.User.Admin))
	fmt.Printf("  Moderator : %s\n", ask.YesNo(this.User.Moderator))
	return nil
}

// ActionShowUser handles the `SHOW USER` command.
func ActionShowUser(cmd *dclish.Command) error {
	// Parse the options.
	login := this.User.Login
	if len(cmd.Args) == 1 {
		if this.User.Admin == 0 {
			fmt.Println("ERROR: You are not an admin.")
			return nil
		}
		login = strings.ToUpper(cmd.Args[0])
	}
	if cmd.Flags["/ALL"].Set {
		if this.User.Admin == 0 {
			fmt.Println("ERROR: You are not an admin.")
			return nil
		}
		if len(cmd.Args) == 1 {
			fmt.Println("ERROR: You can't specify a user with /ALL.")
			return nil
		}
	}
	var showLogin int64
	if cmd.Flags["/LOGIN"].Value == "true" {
		if this.User.Admin == 0 {
			fmt.Println("ERROR: You are not an admin.")
			return nil
		}
		showLogin = 1
	}
	folder := this.Folder
	if cmd.Flags["/FOLDER"].Set {
		fmt.Printf("%v\n", cmd.Flags["/FOLDER"].Value)
		if cmd.Flags["/FOLDER"].Value != "" {
			folder = folders.FindFolder(cmd.Flags["/FOLDER"].Value)
			if folder.Name == "" {
				fmt.Println("ERROR: Folder does not exist.")
				return nil
			}
		}
	}
	/*
		TODO: need to add this.
		var since time.Time
		var err error
		if cmd.Flags["/SINCE"].Value != "" {
			if cmd.Flags["/LOGIN"].Set || cmd.Flags["/FOLDER"].Set {
				fmt.Println("ERROR: Must set /[NO]LOGIN or /FOLDER.")
				return nil
			}
			since, err = ParseDate(cmd.Flags["/SINCE"].Value)
			if err != nil {
				fmt.Println("ERROR: Invalid date specified.")
				return nil
			}
		}
	*/

	// Actually do the thing.
	ctx := storage.Context()
	if cmd.Flags["/LOGIN"].Set && cmd.Flags["/FOLDER"].Set {
		rows, err := this.Q.GetLastReadByEnabled(ctx, folder.Name, showLogin)
		if err != nil {
			fmt.Printf("ERROR: Failed to get list (%s).\n", err)
			return nil
		}
		fmt.Println("User          Folder                     Message #")
		for _, r := range rows {
			fmt.Printf("%-12s  %-25s  % 4d\n", r.Author, folder.Name, r.ID)
		}
	} else if cmd.Flags["/ALL"].Set && cmd.Flags["/FOLDER"].Set {
		rows, err := this.Q.GetLastRead(ctx, folder.Name)
		if err != nil {
			fmt.Printf("ERROR: Failed to get list (%s).\n", err)
			return nil
		}
		fmt.Println("User          Folder                     Message #")
		for _, r := range rows {
			fmt.Printf("%-12s  %-25s  % 4d\n", r.Author, folder.Name, r.ID)
		}
	} else if cmd.Flags["/FOLDER"].Set {
		r, err := this.Q.GetLastReadByUser(ctx, folder.Name, login)
		if err != nil {
			fmt.Printf("ERROR: Failed to get list (%s).\n", err)
			return nil
		}
		fmt.Println("User          Folder                     Message #")
		fmt.Printf("%-12s  %-25s  % 4d\n", r.Author, folder.Name, r.ID)
	} else if cmd.Flags["/LOGIN"].Set {
		rows, err := this.Q.GetLastLoginByEnabled(ctx, showLogin)
		if err != nil {
			fmt.Printf("ERROR: Failed to get list (%s).\n", err)
			return nil
		}
		fmt.Println("User          Last Login")
		for _, r := range rows {
			fmt.Printf("%-12s  %s\n", r.Login, r.LastLogin.Format("2006-01-02 15:04:05"))
		}
	} else if cmd.Flags["/ALL"].Set {
		rows, err := this.Q.GetLastLogin(ctx)
		if err != nil {
			fmt.Printf("ERROR: Failed to get list (%s).\n", err)
			return nil
		}
		fmt.Println("User          Last Login")
		for _, r := range rows {
			fmt.Printf("%-12s  %s\n", r.Login, r.LastLogin.Format("2006-01-02 15:04:05"))
		}
	} else {
		r, err := this.Q.GetLastLoginByLogin(ctx, login)
		if err != nil {
			fmt.Printf("ERROR: Failed to get list (%s).\n", err)
			return nil
		}
		fmt.Println("User          Last Login")
		fmt.Printf("%-12s  %s\n", r.Login, r.LastLogin.Format("2006-01-02 15:04:05"))
	}
	return nil
}

// ActionShowVersion handles the `SHOW VERSION` command.
func ActionShowVersion(_ *dclish.Command) error {
	rev := versioninfo.Revision
	if len(rev) > 7 {
		rev = rev[len(rev)-7:]
	}
	fmt.Printf("BULLETIN Version 2.0.7g (%s)\n", rev)
	fmt.Printf("Linked on %s\n",
		versioninfo.LastCommit.Format("2006-05-04 15:02:01"))
	return nil
}
