package storage

import (
	"context"
	"database/sql"
	"embed"
	"errors"
	"os"
	"path"
	"strings"

	// Included to connect to sqlite.
	_ "github.com/golang-migrate/migrate/v4/database/sqlite"
	_ "modernc.org/sqlite"

	"github.com/adrg/xdg"
	"github.com/golang-migrate/migrate/v4"
	"github.com/golang-migrate/migrate/v4/source/iofs"
	"github.com/jmoiron/sqlx"
)

//go:embed migrations/*.sql
var migrationsFS embed.FS

var dbfile string

// ConfigureTestDBParam has the test functions we need.
type ConfigureTestDBParam interface {
	Helper()
	Fatalf(string, ...any)
}

// ConfigureTestDB configures a test db.
func ConfigureTestDB(t ConfigureTestDBParam) {
	t.Helper()
	tmpFile, err := os.CreateTemp("", "bulletin-test-*.db")
	if err != nil {
		t.Fatalf("failed to create temp file: %v", err)
	}
	tmpFile.Close()

	dbfile = tmpFile.Name()
	db, err := sql.Open("sqlite", dbfile)
	if err != nil {
		os.Remove(dbfile)
		t.Fatalf("failed to open sqlite db: %v", err)
	}
	if err := db.Close(); err != nil {
		os.Remove(dbfile)
		t.Fatalf("failed to close sqlite db: %v", err)
	}
}

// RemoveTestDB configures a test db.
func RemoveTestDB() {
	if strings.Contains(dbfile, "bulletin-test-") {
		os.Remove(dbfile)
	}
}

// Open opens the bulletin database.
func Open() (*sqlx.DB, error) {
	if dbfile == "" {
		// Determine path names and create components.
		bulldir := path.Join(xdg.DataHome, "BULLETIN")
		err := os.MkdirAll(bulldir, 0700)
		if err != nil {
			return nil, errors.New("bulletin directory problem")
		}
		dbfile = path.Join(bulldir, "bulletin.db")
	}

	// Run db migrations if needed.
	migrations, err := iofs.New(migrationsFS, "migrations")
	if err != nil {
		return nil, err
	}
	m, err := migrate.NewWithSourceInstance("iofs", migrations,
		"sqlite://"+dbfile+"?_pragma=foreign_keys(1)")
	if err != nil {
		return nil, err
	}
	defer m.Close()
	err = m.Up()
	if err != nil && err != migrate.ErrNoChange {
		return nil, err
	}

	// Connect to the db.
	db, err := sqlx.Connect("sqlite", "file://"+dbfile+"?_pragma=foreign_keys(1)")
	if err != nil {
		return nil, err
	}

	// Prepare to watch for schema skew - see version.go.
	InitialiseSkewChecker(db, m)

	return db, nil
}

// Context the standard context for storage calls.
func Context() context.Context {
	return context.TODO()
}
