package storage

import (
	"fmt"
	"strings"
	"time"
)

// Alert values.
const (
	AlertDoNotCare int64 = iota
	AlertNone
	AlertBrief
	AlertReadNew
	AlertShowNew
	AlertBriefPerm
	AlertReadNewPerm
	AlertShowNewPerm
)

// EffectiveAlert returns the effective alert from a user and folder alert level.
func EffectiveAlert(user, folder int64) int64 {
	// Folder permanent alert is set - it wins.
	if folder > 4 {
		return folder - 3
	}
	// User doesn't care.
	if user == AlertDoNotCare {
		return folder
	}
	// User's choice of alert level wins.
	return user
}

// AlertString translates an alert number to a string.
func AlertString(alert int64) string {
	var as = []string{
		"Not Set",
		"None",
		"Brief",
		"ReadNew",
		"ShowNew",
	}
	if alert < 0 || alert >= int64(len(as)) {
		return "Unknown"
	}
	return as[alert]
}

// FolderAlertString translates an alert number to a string.
func FolderAlertString(alert int64) string {
	var as = []string{
		"Not Set",
		"None",
		"Brief",
		"ReadNew",
		"ShowNew",
		"Brief (permanent)",
		"ReadNew (permanent)",
		"ShowNew (permanent)",
	}
	if alert < 0 || alert >= int64(len(as)) {
		return "Unknown"
	}
	return as[alert]
}

// String renders a message.
func (m *Message) String() string {
	buf := &strings.Builder{}
	changed := "*"
	if m.CreateAt.Compare(m.UpdateAt) == 0 {
		changed = ""
	}
	fmt.Fprintf(buf, "Msg : %d\n", m.ID)
	fmt.Fprintf(buf, "From: \"%s\" %s%s\n", m.Author,
		m.CreateAt.Format("02-JAN-2006 15:04:05"), changed)
	fmt.Fprintf(buf, "To  : %s\n", m.Folder)
	fmt.Fprintf(buf, "Subj: %s\n\n", m.Subject)
	fmt.Fprintf(buf, "%s\n", m.Message)

	return buf.String()
}

// OneLine renders a message in a line.
func (m *Message) OneLine(expire bool) string {
	var t time.Time
	if expire {
		t = m.Expiration
	} else {
		t = m.CreateAt
	}
	ts := t.Format("2006-01-02 15:04:05")
	return fmt.Sprintf("%4d %-43.43s %-12.12s %-10s\n",
		m.ID, m.Subject, m.Author, ts)
}

// AlertLine renders a message in a line for a SHOWNEW setting.
func (m *Message) AlertLine() string {
	ts := m.CreateAt.Format("2006-01-02 15:04:05")
	return fmt.Sprintf("%-25.25s %4d %-23.23s %-12.12s %-10s\n",
		m.Folder, m.ID, m.Subject, m.Author, ts)
}

// String displays a user (mainly used for debugging).
func (u User) String() string {
	return fmt.Sprintf("%-12s %-25.25s [a%d, m%d, d%d, p%d] [%s]",
		u.Login,
		u.Name,
		u.Admin,
		u.Moderator,
		u.Disabled,
		u.Prompt,
		u.LastLogin.Format("06-01-02 15:04:05"))
}

// String displays a folder (mainly used for debugging).
func (f Folder) String() string {
	return fmt.Sprintf("Folder %s (%s) [a%d, !<%s>, sys%d, exp%d, v%d]",
		f.Name,
		f.Description,
		f.Always,
		FolderAlertString(f.Alert),
		f.System,
		f.Expire,
		f.Visibility)
}

// String displays a folder (mainly used for debugging).
func (b Broadcast) String() string {
	bell := ""
	if b.Bell == 1 {
		bell = "\a\a"
	}
	return fmt.Sprintf("%sFrom: %s   %s\n%s\n",
		bell,
		b.Author,
		b.CreateAt.Format("06-01-02 15:04:05"),
		b.Message)
}
