CREATE TABLE users (
  login          VARCHAR(12)  PRIMARY KEY NOT NULL,
  name           VARCHAR(53)  NOT NULL,
  admin          INT          DEFAULT 0 NOT NULL,
  moderator      INT          DEFAULT 0 NOT NULL,
  disabled       INT          DEFAULT 0 NOT NULL,
  prompt         INT          DEFAULT 0 NOT NULL,
  signature      TEXT         DEFAULT "" NOT NULL,
  last_activity  TIMESTAMP    DEFAULT CURRENT_TIMESTAMP NOT NULL,
  last_login     TIMESTAMP    DEFAULT CURRENT_TIMESTAMP NOT NULL,
  create_at      TIMESTAMP    DEFAULT CURRENT_TIMESTAMP NOT NULL,
  update_at      TIMESTAMP    DEFAULT CURRENT_TIMESTAMP NOT NULL
) WITHOUT ROWID;

CREATE TRIGGER users_after_update_update_at
  AFTER UPDATE ON users FOR EACH ROW
  WHEN NEW.update_at = OLD.update_at    --- avoid infinite loop
BEGIN
  UPDATE users SET update_at=CURRENT_TIMESTAMP WHERE login=NEW.login;
END;

CREATE TRIGGER users_before_update_protect
  AFTER UPDATE ON users FOR EACH ROW
  WHEN OLD.login = 'SYSTEM' AND (NEW.login != OLD.login OR NEW.admin != 1)
BEGIN
  SELECT RAISE (ABORT, 'SYSTEM user is protected');
END;

CREATE TRIGGER users_before_delete_protect
  BEFORE DELETE ON users FOR EACH ROW
  WHEN OLD.login = 'SYSTEM'
BEGIN
  SELECT RAISE (ABORT, 'SYSTEM user is protected');
END;

CREATE TABLE folders (
  name        VARCHAR(25)  PRIMARY KEY NOT NULL,
  always      INT          DEFAULT 0 NOT NULL,
  --- 0=don't care, 1=no, 2(5)=brief(p), 3(6)=readnew(p), 4(7)=shownew(p)
  alert       INT          DEFAULT 1 NOT NULL,
  description VARCHAR(53)  DEFAULT 0 NOT NULL,
  owner       VARCHAR(12)  NOT NULL,
  system      INT          DEFAULT 0 NOT NULL,
  expire      INT          DEFAULT 14 NOT NULL,
  --- public=0, semiprivate=1, private=2
  visibility  INT          DEFAULT 0 NOT NULL,
  create_at   TIMESTAMP    DEFAULT CURRENT_TIMESTAMP NOT NULL,
  update_at   TIMESTAMP    DEFAULT CURRENT_TIMESTAMP NOT NULL
) WITHOUT ROWID;

CREATE TRIGGER folders_before_insert_validate
  BEFORE INSERT on folders
BEGIN
  SELECT
    CASE
      WHEN NEW.name != UPPER(NEW.name) OR NEW.name GLOB '*[^A-Z0-9_-]*' THEN
        RAISE (ABORT, 'Invalid folder name')
    END;
END;

CREATE TRIGGER folders_before_update_validate
  BEFORE UPDATE on folders
BEGIN
  SELECT
    CASE
      WHEN NEW.name != UPPER(NEW.name) OR NEW.name GLOB '*[^A-Z0-9_-]*' THEN
        RAISE (ABORT, 'Invalid folder name')
      WHEN OLD.name = 'GENERAL' AND OLD.name != NEW.name THEN
        RAISE (ABORT, 'GENERAL folder is protected')
    END;
END;

CREATE TRIGGER folders_after_update_update_at
  AFTER UPDATE ON folders FOR EACH ROW
    WHEN NEW.update_at = OLD.update_at    --- avoid infinite loop
BEGIN
  UPDATE folders SET update_at=CURRENT_TIMESTAMP WHERE name=NEW.name;
END;

CREATE TRIGGER folders_before_delete_protect
  BEFORE DELETE on folders FOR EACH ROW
  WHEN OLD.name = 'GENERAL'
BEGIN
  SELECT RAISE (ABORT, 'GENERAL folder is protected');
END;

CREATE TABLE messages (
  id          INT          NOT NULL,
  folder      VARCHAR(25)  REFERENCES folders(name)
                           ON DELETE CASCADE ON UPDATE CASCADE
                           NOT NULL,
  author      VARCHAR(25)  NOT NULL,
  subject     VARCHAR(53)  NOT NULL,
  message     TEXT         NOT NULL,
  permanent   INT          DEFAULT 0 NOT NULL,
  system      INT          DEFAULT 0 NOT NULL,
  shutdown    INT          DEFAULT 0 NOT NULL,
  expiration  TIMESTAMP    NOT NULL,
  create_at   TIMESTAMP    DEFAULT CURRENT_TIMESTAMP NOT NULL,
  update_at   TIMESTAMP    DEFAULT CURRENT_TIMESTAMP NOT NULL,
  PRIMARY KEY (id, folder)
) WITHOUT ROWID;
CREATE INDEX messages_idx_shutdown ON messages(shutdown);
CREATE INDEX messages_idx_expiration ON messages(expiration);

CREATE TRIGGER messages_after_update_update_at
  AFTER UPDATE ON messages FOR EACH ROW
    WHEN NEW.update_at = OLD.update_at    --- avoid infinite loop
BEGIN
  UPDATE messages SET update_at=CURRENT_TIMESTAMP
    WHERE folder=NEW.folder AND id=NEW.id;
END;

CREATE TABLE seen (
  login       VARCHAR(25) REFERENCES users(login)
              ON DELETE CASCADE ON UPDATE CASCADE NOT NULL,
  folder      VARCHAR(25) REFERENCES folders(name)
              ON DELETE CASCADE ON UPDATE CASCADE NOT NULL,
  msgid       INT     NOT NULL,
  create_at   TIMESTAMP    DEFAULT CURRENT_TIMESTAMP NOT NULL,
  --- Nothing to update in this table.
  PRIMARY KEY (folder, login, msgid),
  CONSTRAINT read_fk_id_folder
    FOREIGN KEY (msgid, folder)
    REFERENCES messages(id, folder)
    ON DELETE CASCADE
    ON UPDATE CASCADE
) WITHOUT ROWID;

CREATE TABLE marks (
  login       VARCHAR(25) REFERENCES users(login)
              ON DELETE CASCADE ON UPDATE CASCADE NOT NULL,
  folder      VARCHAR(25) REFERENCES folders(name)
              ON DELETE CASCADE ON UPDATE CASCADE NOT NULL,
  msgid       INT     NOT NULL,
  PRIMARY KEY (folder, login, msgid),
  CONSTRAINT mark_fk_id_folder
    FOREIGN KEY (msgid, folder)
    REFERENCES messages(id, folder)
    ON DELETE CASCADE
    ON UPDATE CASCADE
) WITHOUT ROWID;

CREATE TABLE folder_access (
  login       VARCHAR(25) REFERENCES users(login)
              ON DELETE CASCADE ON UPDATE CASCADE NOT NULL,
  folder      VARCHAR(25) REFERENCES folders(name)
              ON DELETE CASCADE ON UPDATE CASCADE NOT NULL,
  --- read-only=1, read-write=2
  visibility  INT    DEFAULT 1 NOT NULL,
  PRIMARY KEY (login, folder)
) WITHOUT ROWID;

--- User folder configs.
CREATE TABLE folder_configs (
  login       VARCHAR(25) REFERENCES users(login)
              ON DELETE CASCADE ON UPDATE CASCADE NOT NULL,
  folder      VARCHAR(25) REFERENCES folders(name)
              ON DELETE CASCADE ON UPDATE CASCADE NOT NULL,
  always      INT     DEFAULT 0 NOT NULL,
  --- 0=don't care, 1=no, 2=brief, 3=readnew, 4=shownew
  alert       INT     DEFAULT 0 NOT NULL,
  PRIMARY KEY (login, folder)
) WITHOUT ROWID;

--- System configs.
CREATE TABLE system (
  name            VARCHAR(12)  PRIMARY KEY NOT NULL,
  default_expire  INT          DEFAULT -1 NOT NULL,
  expire_limit    INT          DEFAULT -1 NOT NULL
);

--- Broadcast messages.
CREATE TABLE broadcast (
  author          VARCHAR(12)  PRIMARY KEY NOT NULL,
  bell            INT          DEFAULT 0 NOT NULL,
  message         TEXT         NOT NULL,
  create_at       TIMESTAMP    DEFAULT CURRENT_TIMESTAMP NOT NULL
);
