-- CreateMessage creates a new message.
-- name: CreateMessage :exec
INSERT INTO messages (
  id, folder, author, subject, message, permanent, shutdown, expiration
) VALUES (
  (SELECT COALESCE(MAX(id), 0) + 1 FROM messages AS m WHERE m.folder = ?1),
  ?1, ?2, ?3, ?4, ?5, ?6, ?7);

-- SetMessageSeen mark a message as seen for a user.
-- name: SetMessageSeen :exec
INSERT INTO seen (login, folder, msgid) VALUES (?, ?, ?);

-- UnsetMessageSeen unmark a message as seen for a user.
-- name: UnsetMessageSeen :exec
DELETE FROM seen WHERE login = ? AND folder = ? AND msgid = ?;

-- ReadMessage returns a specified message id from a folder.
-- name: ReadMessage :one
SELECT * FROM messages WHERE folder = ? AND id = ?;

-- NextMsgid gets the next unread message id.
-- name: NextMsgid :one
SELECT CAST(COALESCE(MIN(id), 0) AS INT) FROM messages AS m
  WHERE m.folder = ?1 AND m.id > ?2
  AND id NOT IN (SELECT msgid FROM seen AS s WHERE s.folder = ?1 AND s.login = ?3);

-- NextMsgidIgnoringSeen gets the next message id.
-- name: NextMsgidIgnoringSeen :one
SELECT CAST(MIN(id) AS INT) FROM messages AS m
  WHERE m.folder = ?1 AND m.id > ?2;

-- PrevMsgid get the previous message id.
-- name: PrevMsgid :one
SELECT CAST(COALESCE(MAX(id), 0) AS INT) FROM messages AS m
  WHERE m.folder = ?1 AND m.id < ?2
  AND id NOT IN (SELECT msgid FROM seen AS s WHERE s.folder = ?1 AND s.login = ?3);

-- DeleteAllMessages delete all messages in a folder.
-- name: DeleteAllMessages :exec
DELETE FROM messages WHERE folder = ?;

-- LastMsgidIgnoringSeen last message id in a folder without factoring in
-- if the message has been seen.
-- name: LastMsgidIgnoringSeen :one
SELECT CAST(MAX(id) AS INT) FROM messages AS m WHERE m.folder = ?1;

-- GetLastRead gets the last message read by a login in a folder.
-- name: GetLastRead :many
SELECT CAST(MAX(m.id) AS INT) AS id, m.author FROM messages AS m, users AS u
  WHERE folder = ? AND u.login == m.author
  GROUP BY m.author
  ORDER BY m.author;

-- GetLastReadByEnabled gets the last message for a user that's enabled.
-- name: GetLastReadByEnabled :many
SELECT CAST(MAX(m.id) AS INT) AS id, m.author FROM messages AS m, users AS u
  WHERE folder = ? AND u.login == m.author AND u.disabled = ?
  GROUP BY m.author
  ORDER BY m.author;

-- GetLastReadByUser get last message read by a user in a folder.
-- name: GetLastReadByUser :one
SELECT CAST(MAX(m.id) AS INT) AS id, m.author FROM messages AS m, users AS u
  WHERE folder = ? AND u.login == m.author AND m.author = ?;

-- Search find messages that match subject or message text.
-- name: Search :many
SELECT * FROM messages
  WHERE (message LIKE '%' || ?1 || '%'
     OR subject LIKE '%' || ?1 || '%')
    AND id >= ?2
    AND folder = ?3
  ORDER BY id ASC;

-- SearchReverse find messages that match subject or message text in reverse order.
-- name: SearchReverse :many
SELECT * FROM messages
  WHERE (message LIKE '%' || ?1 || '%'
     OR subject LIKE '%' || ?1 || '%')
    AND id <= ?2
    AND folder = ?3
  ORDER BY id DESC;

-- SearchSubject find messages that match the subject.
-- name: SearchSubject :many
SELECT * FROM messages
  WHERE subject LIKE '%' || ? || '%'
    AND id >= ?
    AND folder = ?
  ORDER BY id ASC;

-- SearchSubjectReverse find messages that match the subject in reverse order.
-- name: SearchSubjectReverse :many
SELECT * FROM messages
  WHERE subject LIKE '%' || ? || '%'
    AND id <= ?
    AND folder = ?
  ORDER BY id DESC;

-- SearchReply find messages that match a specific subject and a different
-- message text.
-- name: SearchReply :many
SELECT * FROM messages
  WHERE message LIKE '%' || ? || '%'
    AND subject = ?
    AND id >= ?
    AND folder = ?
  ORDER BY id ASC;

-- SearchReplyReverse find messages that match a specific subject and
-- a different  message text in reverse order.
-- name: SearchReplyReverse :many
SELECT * FROM messages
  WHERE message LIKE '%' || ? || '%'
    AND subject = ?
    AND id <= ?
    AND folder = ?
  ORDER BY DESC;

-- UpdateMessage updates a message.
-- name: UpdateMessage :exec
UPDATE messages SET
    subject = ?,
    message = ?,
    permanent = ?,
    system = ?,
    shutdown = ?,
    expiration = ?
  WHERE id = ?;

-- GetAlertMessages get all alert messages for a user older than a
-- certain date that are brief, readnew, or shownew.
-- name: GetAlertMessages :many
SELECT CAST(f.alert AS INT) AS folder_alert,
       CAST(COALESCE(fc.alert, 0) AS INT) AS user_alert,
       sqlc.embed(m)
  FROM messages AS m
  LEFT OUTER JOIN folders AS f ON m.folder = f.name
  LEFT OUTER JOIN folder_configs AS fc ON f.name = fc.folder AND fc.login = ?
  WHERE m.create_at >= ?
    AND (f.alert > 4 OR fc.alert > 1
         OR (f.alert > 1 AND COALESCE(fc.alert, 0) != 1))
  ORDER BY m.folder, m.id;
